/*
	File			:	Q9AX.CPP
	Identification	:	9 nodes, quadrilateral element, axisymmetry
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	15.07.2006
	Last modified	:	07.03.2009
	Content			:	NQ9, CenterQ9, GaussQ9, mkQ9, mkBQ9, KmQ9, esQ9, rlQ9, N9
*/

#ifndef _srokosz_Q9AX_cpp
#define _srokosz_Q9AX_cpp

/* BEGIN Q9 */

#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include "FEMVars.h"
#include "FEMLibAX.h"
#include "MCPLAX.H"
#include "MMC.H"

static double B[4][18];

//-------------------------------------------------------------------------------------
//shape functions for Q9
COOR4 NQ9( int iGPoint )
{
	COOR4 N;
	double ksi, eta;
	
	ksi = GPQ8[iGPoint].ksi;
	eta = GPQ8[iGPoint].eta;

	N.one = Quarter * ksi * ( ksi - One ) * eta * ( eta - One );
	N.two = NHalf * ksi * ( ksi - One ) * ( eta + One ) * ( eta - One );
	N.three = Quarter * ksi * ( ksi - One ) * eta * ( eta + One );
	N.four = NHalf * ( ksi + One ) * ( ksi - One ) * eta * ( eta + One );
	N.five = Quarter * ksi * ( ksi + One ) * eta * ( eta + One );
	N.six = NHalf * ksi * ( ksi + One ) * ( eta + One ) * ( eta - One );
	N.seven = Quarter * ksi * ( ksi + One ) * eta * ( eta - One );
	N.eight = NHalf * ( ksi + One )* ( ksi - One ) * eta * ( eta - One );
	N.nine = ( ksi + One )* ( ksi - One ) * ( eta + One ) * ( eta - One );
	return N;

}//NQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Center point for Q9 element in global coordinates
//used for material constants identification
COOR2 CenterQ9( long iElem )
{
	COOR2 G, P1, P2, P3, P4, P5, P6, P7, P8, P9;
	COOR4 N;
	double ksi, eta;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	P5 = NODE[ELEM[iElem].node[4]];
	P6 = NODE[ELEM[iElem].node[5]];
	P7 = NODE[ELEM[iElem].node[6]];
	P8 = NODE[ELEM[iElem].node[7]];
	P9 = NODE[ELEM[iElem].node[8]];
	
	ksi = Zero;
	eta = Zero;

	N.one = Quarter * ksi * ( ksi - One ) * eta * ( eta - One );
	N.two = NHalf * ksi * ( ksi - One ) * ( eta + One ) * ( eta - One );
	N.three = Quarter * ksi * ( ksi - One ) * eta * ( eta + One );
	N.four = NHalf * ( ksi + One ) * ( ksi - One ) * eta * ( eta + One );
	N.five = Quarter * ksi * ( ksi + One ) * eta * ( eta + One );
	N.six = NHalf * ksi * ( ksi + One ) * ( eta + One ) * ( eta - One );
	N.seven = Quarter * ksi * ( ksi + One ) * eta * ( eta - One );
	N.eight = NHalf * ( ksi + One )* ( ksi - One ) * eta * ( eta - One );
	N.nine = ( ksi + One )* ( ksi - One ) * ( eta + One ) * ( eta - One );

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x;
	G.x += N.five * P5.x + N.six * P6.x + N.seven * P7.x + N.eight * P8.x + N.nine * P9.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y;
	G.y += N.five * P5.y + N.six * P6.y + N.seven * P7.y + N.eight * P8.y + N.nine * P9.y;
	return G;

}//CenterQ9
//-------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------
//transformation : local to global coordinates
// used for Gauss integration points: stress&strains location
COOR2 GaussQ9( long iElem, int iGPoint )
{
	COOR2 G, P1, P2, P3, P4, P5, P6, P7, P8, P9;
	COOR4 N;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	P5 = NODE[ELEM[iElem].node[4]];
	P6 = NODE[ELEM[iElem].node[5]];
	P7 = NODE[ELEM[iElem].node[6]];
	P8 = NODE[ELEM[iElem].node[7]];
	P9 = NODE[ELEM[iElem].node[8]];
	
	N = NQ9( iGPoint );

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x;
	G.x += N.five * P5.x + N.six * P6.x + N.seven * P7.x + N.eight * P8.x + N.nine * P9.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y;
	G.y += N.five * P5.y + N.six * P6.y + N.seven * P7.y + N.eight * P8.y + N.nine * P9.y;
	return G;

}//GaussQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void mkQ9( long iElem, long iNode1, long iNode2, long iNode3, long iNode4, long iNode5,
		    long iNode6, long iNode7, long iNode8, long iNode9 )
{
	int i;
	
	NELEM++;
	ELEM[iElem].typ = 9;
	ELEM[iElem].node[0] = iNode1;
	ELEM[iElem].node[1] = iNode2;
	ELEM[iElem].node[2] = iNode3;
	ELEM[iElem].node[3] = iNode4;
	ELEM[iElem].node[4] = iNode5;
	ELEM[iElem].node[5] = iNode6;
	ELEM[iElem].node[6] = iNode7;
	ELEM[iElem].node[7] = iNode8;
	ELEM[iElem].node[8] = iNode9;
	ELEM[iElem].matpars = MATPAR_ZERO;
	for( i=0L; i<9L; i++ )
	{
		ELEM[iElem].Fmc[i] = NOne;
		ELEM[iElem].eps[i] = COOR3_ZERO;
		ELEM[iElem].sig[i] = COOR3_ZERO;
		ELEM[iElem].dsig[i] = COOR3_ZERO;
		ELEM[iElem].deps[i] = COOR3_ZERO;
		ELEM[iElem].depsvp[i] = COOR3_ZERO;
		ELEM[iElem].dsigvp[i] = COOR3_ZERO;
		ELEM[iElem].dQ[i] = COOR3_ZERO;
		ELEM[iElem].Gauss[i] = COOR2_ZERO;
		ELEM[iElem].u[i] = Zero;
		ELEM[iElem].plas[i] = 0L;
		ELEM[iElem].Edp[i]  = Zero;
	}
	//consolidation stress
	for( i=0L; i<9L; i++ )
	{
		ELEM[iElem].sig[i].x = consolid_stress;
		ELEM[iElem].sig[i].y = consolid_stress;
		ELEM[iElem].sig[i].z = consolid_stress;
	}

}//mkQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Q9 element - B matrix
double mkBQ9( long iE, long iG )
{
	double J[2][2], InvJ[2][2], DetJ;
	double ksi, eta;
	double dN[2][9], b[2][9], ksieta, ksi2, eta2;
	long row, col, i;
	COOR2 P1, P2, P3, P4, P5, P6, P7, P8, P9, r;
	COOR4 Ni;

	for( row=0; row<4; row++ )
	{
		for( col=0; col<18; col++ )
			B[row][col] = Zero;
	}

	P1 = NODE[ELEM[iE].node[0]];
	P2 = NODE[ELEM[iE].node[1]];
	P3 = NODE[ELEM[iE].node[2]];
	P4 = NODE[ELEM[iE].node[3]];
	P5 = NODE[ELEM[iE].node[4]];
	P6 = NODE[ELEM[iE].node[5]];
	P7 = NODE[ELEM[iE].node[6]];
	P8 = NODE[ELEM[iE].node[7]];
	P9 = NODE[ELEM[iE].node[8]];

	/*
	y
	|
	|       P3
	|	  /    \
	|	P1  P9  P5
	|	 \      /
	|	   \  /
	|	    P7
	 ------------------x

	eta
  P3  |  P5
	0---0
	| | |
	| P9-------ksi
	|   |
	0---0	
  P1     P7
	*/

	//axisymmetry
	r = GaussQ9( iE, iG );
	Ni = NQ9( iG );

	//coordinates of Gauss point
	ksi = GPQ8[iG].ksi;
	eta = GPQ8[iG].eta;
		
	ksieta = ksi * eta;
	ksi2 = ksi * ksi;
	eta2 = eta * eta;

	//derivatives dN/dlocal
	dN[0][0] = Quarter * eta * ( Two * ksieta - Two * ksi - eta + One );
	dN[0][1] = NHalf * ( Two * ksieta * eta - Two * ksi - eta2 + One );
	dN[0][2] = Quarter * eta * ( Two * ksieta + Two * ksi - eta - One );
	dN[0][3] = NOne * ksieta * ( eta + One );
	dN[0][4] = Quarter * eta * ( Two * ksieta + Two * ksi + eta + One );
	dN[0][5] = NHalf * ( Two * ksieta * eta - Two * ksi + eta2 - One );
	dN[0][6] = Quarter * eta * ( Two * ksieta - Two * ksi + eta - One );
	dN[0][7] = ksieta * ( One - eta );
	dN[0][8] = Two * ksi * ( eta2 - One );
	dN[1][0] = Quarter * ksi * ( Two * ksieta - ksi - Two * eta + One );
	dN[1][1] = ksieta * ( One - ksi );
	dN[1][2] = Quarter * ksi * ( Two * ksieta + ksi - Two * eta - One );
	dN[1][3] = NHalf * ( Two * ksi * ksieta + ksi2 - Two * eta - One );
	dN[1][4] = Quarter * ksi * ( Two * ksieta + ksi + Two * eta + One );
	dN[1][5] = NOne * ksieta * ( ksi + One );
	dN[1][6] = Quarter * ksi * ( Two * ksieta - ksi + Two * eta - One );
	dN[1][7] = NHalf * ( Two * ksieta * ksi - ksi2 - Two * eta + One );
	dN[1][8] = Two * eta * ( ksi2 - One );

	// Jacobi matrix = dN/dLocal * P(coor)
	J[0][0] = dN[0][0] * P1.x + dN[0][1] * P2.x + dN[0][2] * P3.x + dN[0][3] * P4.x;
	J[0][0] += dN[0][4] * P5.x + dN[0][5] * P6.x + dN[0][6] * P7.x + dN[0][7] * P8.x;
	J[0][0] += dN[0][8] * P9.x;
	J[0][1] = dN[0][0] * P1.y + dN[0][1] * P2.y + dN[0][2] * P3.y + dN[0][3] * P4.y;
	J[0][1] += dN[0][4] * P5.y + dN[0][5] * P6.y + dN[0][6] * P7.y + dN[0][7] * P8.y;
	J[0][1] += dN[0][8] * P9.y;
	J[1][0] = dN[1][0] * P1.x + dN[1][1] * P2.x + dN[1][2] * P3.x + dN[1][3] * P4.x;
	J[1][0] += dN[1][4] * P5.x + dN[1][5] * P6.x + dN[1][6] * P7.x + dN[1][7] * P8.x;
	J[1][0] += dN[1][8] * P9.x;
	J[1][1] = dN[1][0] * P1.y + dN[1][1] * P2.y + dN[1][2] * P3.y + dN[1][3] * P4.y;
	J[1][1] += dN[1][4] * P5.y + dN[1][5] * P6.y + dN[1][6] * P7.y + dN[1][7] * P8.y;
	J[1][1] += dN[1][8] * P9.y;
			
	//determinant of Jacobi matrix - Jacobian
	DetJ = J[0][0] * J[1][1] - J[0][1] * J[1][0];

	//inverted Jacobi matrix
	InvJ[0][0] = J[1][1] / DetJ;
	InvJ[0][1] = NOne * J[0][1] / DetJ;
	InvJ[1][0] = NOne * J[1][0] / DetJ;
	InvJ[1][1] = J[0][0] / DetJ;

	//B matrix
	for( row=0; row<2; row++ )
	{
		for( col=0; col<9; col++ )
		{
			b[row][col] = Zero;
			for( i=0; i<2; i++ )
				b[row][col] += InvJ[row][i] * dN[i][col];
		}
	}
	B[0][0] = b[0][0]; 
	B[0][2] = b[0][1]; 
	B[0][4] = b[0][2]; 
	B[0][6] = b[0][3];
	B[0][8] = b[0][4]; 
	B[0][10] = b[0][5]; 
	B[0][12] = b[0][6]; 
	B[0][14] = b[0][7];
	B[0][16] = b[0][8];
	B[1][1] = b[1][0]; 
	B[1][3] = b[1][1]; 
	B[1][5] = b[1][2];
	B[1][7] = b[1][3]; 
	B[1][9] = b[1][4]; 
	B[1][11] = b[1][5];
	B[1][13] = b[1][6];
	B[1][15] = b[1][7];
	B[1][17] = b[1][8];
	B[2][0] = B[1][1]; 
	B[2][1] = B[0][0]; 
	B[2][2] = B[1][3]; 
	B[2][3] = B[0][2]; 
	B[2][4] = B[1][5]; 
	B[2][5] = B[0][4]; 
	B[2][6] = B[1][7]; 
	B[2][7] = B[0][6]; 
	B[2][8] = B[1][9]; 
	B[2][9] = B[0][8]; 
	B[2][10] = B[1][11]; 
	B[2][11] = B[0][10]; 
	B[2][12] = B[1][13]; 
	B[2][13] = B[0][12]; 
	B[2][14] = B[1][15]; 
	B[2][15] = B[0][14]; 
	B[2][16] = B[1][17]; 
	B[2][17] = B[0][16];
	B[3][0] = Ni.one / r.x;
	B[3][2] = Ni.two / r.x;
	B[3][4] = Ni.three / r.x;
	B[3][6] = Ni.four / r.x;
	B[3][8] = Ni.five / r.x;
	B[3][10] = Ni.six / r.x;
	B[3][12] = Ni.seven / r.x;
	B[3][14] = Ni.eight / r.x;
	B[3][16] = Ni.nine / r.x;
	
	return DetJ;

}//mkQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Q9 element - stiffness
void KmQ9( long iElem )
{
	double BTD[18][4], BTDB[18][18], km[18][18], DetJ, Weight;
	long row, col, i, j, lrow, lcol, ixy, jxy, iGauss;
	COOR2 r;

	//zeroing km
	for( row=0; row<18; row++ )
	{
		for( col=0; col<18; col++ )
			km[row][col] = Zero;
	}
	
	//for each Gauss point
	for( iGauss=0; iGauss<9; iGauss++ )
	{		
		STRESS[0] = ELEM[iElem].sig[iGauss].x;
		STRESS[1] = ELEM[iElem].sig[iGauss].y;
		STRESS[2] = ELEM[iElem].sig[iGauss].xy;
		STRESS[3] = ELEM[iElem].sig[iGauss].z;

		if( DEPuse == 1L )
			mkDEP( ELEM[iElem].plas[iGauss], STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		else
			mkDEP( 0L, STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars  );

		//weight of Gauss point
		Weight = GPQ8[iGauss].w;
		
		//B matrix, DetJ
		DetJ = mkBQ9( iElem, iGauss );

		//stiffnes matrix BTDB	
		//first : multiplication BT*D
		//transposed B in multiplication
		for( row=0; row<18; row++ )
		{
			for( col=0; col<4; col++ )
			{
				BTD[row][col] = Zero;
				for( i=0; i<4; i++ ) 
					BTD[row][col] += B[i][row] * DEP[i][col];
			}
		}

		//second : multiplication BTD*B
		for( row=0; row<18; row++ )
		{
			for( col=0; col<18; col++ )
			{
				BTDB[row][col] = Zero;
				for( i=0; i<4; i++ )
					BTDB[row][col] += BTD[row][i] * B[i][col];
			}
		}
		
		//axisymmetry
		r = GaussQ9( iElem, iGauss );
		
		//third : integration DetJ*Wi*BTDB*r
		for( row=0; row<18; row++ )
		{
			for( col=0; col<18; col++ )
					km[row][col] += DetJ * Weight * BTDB[row][col] * r.x;
		}		
	}

	//update global stiffness matrix
	i = 0;
	for( lrow=0; lrow<9; lrow++ )
	{
		for( ixy=0; ixy<2; ixy++ )
		{
			row = 2 * ELEM[iElem].node[lrow] + ixy;
			j = 0;
			for( lcol=0; lcol<9; lcol++ )
			{
				for( jxy=0; jxy<2; jxy++ )
				{
					col = 2 * ELEM[iElem].node[lcol] + jxy;				
					Km[row][col] += km[i][j];
					j+=1;
				}
			}
			i+=1;
		}
	}	
}//KmQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//stresses & strains Q9
void esQ9( long iElem, int opt )
{
	double uQ9[18], Fc, DetJ, Edp_new, sig[4];
	long col, iGauss;
	COOR2 U;
	COOR3 deps;
	
	//retrieving displacement increments
	U = readNODE( u, ELEM[iElem].node[0] );
	uQ9[0] = U.x;
	uQ9[1] = U.y;
	U = readNODE( u, ELEM[iElem].node[1] );
	uQ9[2] = U.x;
	uQ9[3] = U.y;
	U = readNODE( u, ELEM[iElem].node[2] );
	uQ9[4] = U.x;
	uQ9[5] = U.y;
	U = readNODE( u, ELEM[iElem].node[3] );
	uQ9[6] = U.x;
	uQ9[7] = U.y;
	U = readNODE( u, ELEM[iElem].node[4] );
	uQ9[8] = U.x;
	uQ9[9] = U.y;
	U = readNODE( u, ELEM[iElem].node[5] );
	uQ9[10] = U.x;
	uQ9[11] = U.y;
	U = readNODE( u, ELEM[iElem].node[6] );
	uQ9[12] = U.x;
	uQ9[13] = U.y;
	U = readNODE( u, ELEM[iElem].node[7] );
	uQ9[14] = U.x;
	uQ9[15] = U.y;
	U = readNODE( u, ELEM[iElem].node[8] );
	uQ9[16] = U.x;
	uQ9[17] = U.y;
   
	//for each Gauss point
	for( iGauss=0; iGauss<9; iGauss++ )
	{		
		//B matrix, DetJ (not used here)
		DetJ = mkBQ9( iElem, iGauss );

		//strains increment, de=B*du
		deps.x = Zero;
		deps.y = Zero;
		deps.xy = Zero;
		deps.z = Zero;		
		for( col=0; col<18; col++ ) 
		{
			deps.x  += B[0][col] * uQ9[col];		
		    deps.y  += B[1][col] * uQ9[col];		
		    deps.xy += B[2][col] * uQ9[col];
		    deps.z  += B[3][col] * uQ9[col];
		}		
		
		if( opt == 0 )
		{
			//stress increment (Newton-Raphson)
			Edp_new = calcDSIG( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].dsig[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		
			//updating incremental stresses & strains
			ELEM[iElem].deps[iGauss].x  += deps.x;
			ELEM[iElem].deps[iGauss].y  += deps.y;
			ELEM[iElem].deps[iGauss].xy += deps.xy;
			ELEM[iElem].deps[iGauss].z  += deps.z;
			ELEM[iElem].dsig[iGauss].x  += dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  += dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy += dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  += dSTRESS[3];
			//yield value
			sig[0] = ELEM[iElem].sig[iGauss].x + ELEM[iElem].dsig[iGauss].x;
			sig[1] = ELEM[iElem].sig[iGauss].y + ELEM[iElem].dsig[iGauss].y;
			sig[2] = ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy;
			sig[3] = ELEM[iElem].sig[iGauss].z + ELEM[iElem].dsig[iGauss].z;
			Fc = Fmc_MMC( sig, Fi_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ), c_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ));
		}//N-R
		else
		{
			//viscoplastic algorithm

			//minimum DT, dsig=>dSTRESS, dQ=>DQ 
			Fc = calcVP( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].depsvp[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );

			//saving stresses & strains increments
			ELEM[iElem].deps[iGauss].x  = deps.x;
			ELEM[iElem].deps[iGauss].y  = deps.y;
			ELEM[iElem].deps[iGauss].xy = deps.xy;
			ELEM[iElem].deps[iGauss].z  = deps.z;
			
			ELEM[iElem].dsig[iGauss].x  = dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  = dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy = dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  = dSTRESS[3];

			//saving Q-derivatives
			ELEM[iElem].dQ[iGauss].x  = DQ[0];
			ELEM[iElem].dQ[iGauss].y  = DQ[1];
			ELEM[iElem].dQ[iGauss].xy = DQ[2];
			ELEM[iElem].dQ[iGauss].z  = DQ[3];			
		}//vp
		
		//updating markers
		ELEM[iElem].Fmc[iGauss] = Fc;
		if( Fc > ( NOne * Fmin) ) 
		{
			ELEM[iElem].plas[iGauss] = 1L;
		}
		else
		{
			ELEM[iElem].plas[iGauss] = 0L;
		}

		//pore pressure, u=Du*eps
		//matrix D, U
		mkDU( ELEM[iElem].matpars.Ke );
		ELEM[iElem].u[iGauss] = initial_u;
		ELEM[iElem].u[iGauss] += ELEM[iElem].matpars.Ke * ( ELEM[iElem].eps[iGauss].x + ELEM[iElem].eps[iGauss].y + ELEM[iElem].eps[iGauss].z );
		
		//Gauss point
		ELEM[iElem].Gauss[iGauss] = GaussQ9( iElem, iGauss );		
	}//Gauss point
}//esQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//viscoplastic strain & stress increment
void vpQ9( long iElem )
{
	double ddepsvp[4], dsigvp[4];
	long row, col, iGauss;

	mkDE( ELEM[iElem].matpars.E, ELEM[iElem].matpars.ni, ELEM[iElem].matpars.Ke );
	for( iGauss=0L; iGauss<9L; iGauss++ )
	{
		if( ELEM[iElem].Fmc[iGauss] > Zero)
		{
			//viscoplastic strain increment
			ddepsvp[0] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].x;
			ddepsvp[1] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].y;
			ddepsvp[2] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].xy;
			ddepsvp[3] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].z;

			//plastic deviatoric strain
			ELEM[iElem].Edp[iGauss] += DT * ELEM[iElem].Fmc[iGauss];
			
			//updating viscoplastic strain increment
			ELEM[iElem].depsvp[iGauss].x  += ddepsvp[0];
			ELEM[iElem].depsvp[iGauss].y  += ddepsvp[1];
			ELEM[iElem].depsvp[iGauss].xy += ddepsvp[2];
			ELEM[iElem].depsvp[iGauss].z  += ddepsvp[3];

			//viscoplastic stress increment
			for( row=0L; row<4L; row++ )
			{
				dsigvp[row] = Zero;
				for( col=0L; col<4L; col++ )
				{
					dsigvp[row] += DE[row][col] * ddepsvp[col];
				}
			}
		}//F>0
		else
		{
			for( row=0L; row<4L; row++ ) dsigvp[row] = Zero;
		}
			
		//saving viscoplastic stress increment
		ELEM[iElem].dsigvp[iGauss].x  = dsigvp[0];
		ELEM[iElem].dsigvp[iGauss].y  = dsigvp[1];
		ELEM[iElem].dsigvp[iGauss].xy = dsigvp[2];
		ELEM[iElem].dsigvp[iGauss].z  = dsigvp[3];
	}//iGauss
}//vpQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//body loads, new version as residual loads
void rlQ9( long iElem, int opt )
{
	int iGauss, row;
	double DetJ, BTS[18], fR[18];
	COOR2 r;

	//zeroing
	for( row=0; row<18; row++ ) fR[row] = Zero;
	
	//integration
	for( iGauss=0; iGauss<9; iGauss++ )
	{
		//B matrix
		DetJ = mkBQ9( iElem, iGauss );

		if( opt == 0 )
		{
			//multiplication BT*(sig+dsig)
			for( row=0; row<18; row++ )
			{
				BTS[row]  = B[0][row] * ( ELEM[iElem].sig[iGauss].x  + ELEM[iElem].dsig[iGauss].x );
				BTS[row] += B[1][row] * ( ELEM[iElem].sig[iGauss].y  + ELEM[iElem].dsig[iGauss].y );
				BTS[row] += B[2][row] * ( ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy );
				BTS[row] += B[3][row] * ( ELEM[iElem].sig[iGauss].z  + ELEM[iElem].dsig[iGauss].z );
			}
		}
		else
		{
			//multiplication BT*dsigvp
			for( row=0; row<18; row++ )
			{
				BTS[row]  = B[0][row] * ELEM[iElem].dsigvp[iGauss].x;
				BTS[row] += B[1][row] * ELEM[iElem].dsigvp[iGauss].y;
				BTS[row] += B[2][row] * ELEM[iElem].dsigvp[iGauss].xy;
				BTS[row] += B[3][row] * ELEM[iElem].dsigvp[iGauss].z;
			}
		}
				
		//axisymmetry
		r = GaussQ9( iElem, iGauss );
		
		//integration
		for( row=0; row<18; row++ )
		{
			fR[row] += DetJ * GPQ8[iGauss].w * BTS[row] * r.x;
		}
	}
	
	updatefR( ELEM[iElem].node[0], fR[0], fR[1] );
	updatefR( ELEM[iElem].node[1], fR[2], fR[3] );
	updatefR( ELEM[iElem].node[2], fR[4], fR[5] );
	updatefR( ELEM[iElem].node[3], fR[6], fR[7] );
	updatefR( ELEM[iElem].node[4], fR[8], fR[9] );
	updatefR( ELEM[iElem].node[5], fR[10], fR[11] );
	updatefR( ELEM[iElem].node[6], fR[12], fR[13] );
	updatefR( ELEM[iElem].node[7], fR[14], fR[15] );
	updatefR( ELEM[iElem].node[8], fR[16], fR[17] );

}//rlQ9
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double N9( long iElem, int iNode )
{
	int iGauss;
	double vN, NQ, DetJ;
	COOR2 r;
	
	vN = Zero;
	for(iGauss=0; iGauss<9; iGauss++)
	{
		switch(iNode)
		{
			case 1: NQ = NQ9(iGauss).one;
					break;
			case 2: NQ = NQ9(iGauss).two;
					break;
			case 3:	NQ = NQ9(iGauss).three;
					break;
			case 4:	NQ = NQ9(iGauss).four;
					break;
			case 5: NQ = NQ9(iGauss).five;
					break;
			case 6: NQ = NQ9(iGauss).six;
					break;
			case 7:	NQ = NQ9(iGauss).seven;
					break;
			case 8:	NQ = NQ9(iGauss).eight;
					break;
			case 9:	NQ = NQ9(iGauss).nine;
					break;
		}
		//axisymmetry
		r = GaussQ9( iElem, iGauss );
		DetJ = mkBQ9( iElem, iGauss );
		vN += DetJ * GPQ8[iGauss].w * NQ * r.x;
	}
	return vN;
}//N9
//-------------------------------------------------------------------------------------


/* END Q9 */

#endif