/*
	File			:	Q4AX.CPP
	Identification		:	4 nodes, quadrilateral element, Axisymmetry
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	15.07.2006
	Last modified		:	07.03.2009
	Content			:	NQ4, CenterQ4, GaussQ4, mkQ4, mkBQ4, KmQ4, esQ4, N4
*/

#ifndef _srokosz_Q4AX_cpp
#define _srokosz_Q4AX_cpp

/* BEGIN Q4 */

#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include "FEMVars.h"
#include "FEMLibAX.h"
#include "MCPLAX.H"
#include "MMC.H"
#include <stdio.h>
#include <float.h>

static double B[4][8];

//-------------------------------------------------------------------------------------
//shape functions for Q4
COOR4 NQ4( int iGPoint )
{
	COOR4 N;
	double ksi, eta;
	
	ksi = GPQ4[iGPoint].ksi;
	eta = GPQ4[iGPoint].eta;

	N.one = Quarter * ( One - ksi ) * ( One - eta );
	N.two = Quarter * ( One - ksi ) * ( One + eta );
	N.three = Quarter * ( One + ksi ) * ( One + eta );
	N.four = Quarter * ( One + ksi ) * ( One - eta );
	return N;

}//NQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//center point of Q4 element in global coordinates
//used for material constants identification
COOR2 CenterQ4( long iElem )
{
	COOR2 G, P1, P2, P3, P4;
	COOR4 N;
	double ksi, eta;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	

	ksi = Zero;
	eta = Zero;

	N.one = Quarter * ( One - ksi ) * ( One - eta );
	N.two = Quarter * ( One - ksi ) * ( One + eta );
	N.three = Quarter * ( One + ksi ) * ( One + eta );
	N.four = Quarter * ( One + ksi ) * ( One - eta );

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y;
	return G;

}//CenterQ4
//-------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------
//transformation : local to global coordinates
// used for Gauss integration points: stress&strains location
COOR2 GaussQ4( long iElem, int iGPoint )
{
	COOR2 G, P1, P2, P3, P4;
	COOR4 N;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	
	N = NQ4( iGPoint );

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y;
	return G;

}//GaussQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//making Q4
void mkQ4( long iElem, long iNode1, long iNode2, long iNode3, long iNode4 )
{
	int i;
	
	NELEM++;
	ELEM[iElem].typ = 4;
	ELEM[iElem].node[0] = iNode1;
	ELEM[iElem].node[1] = iNode2;
	ELEM[iElem].node[2] = iNode3;
	ELEM[iElem].node[3] = iNode4;
	ELEM[iElem].matpars = MATPAR_ZERO;
	for( i=0L; i<9L; i++ )
	{
		ELEM[iElem].Fmc[i] = NOne;
		ELEM[iElem].eps[i] = COOR3_ZERO;
		ELEM[iElem].sig[i] = COOR3_ZERO;
		ELEM[iElem].dsig[i] = COOR3_ZERO;
		ELEM[iElem].deps[i] = COOR3_ZERO;
		ELEM[iElem].depsvp[i] = COOR3_ZERO;
		ELEM[iElem].dsigvp[i] = COOR3_ZERO;
		ELEM[iElem].dQ[i] = COOR3_ZERO;
		ELEM[iElem].Gauss[i] = COOR2_ZERO;
		ELEM[iElem].u[i] = Zero;
		ELEM[iElem].plas[i] = 0L;
		ELEM[iElem].Edp[i]  = Zero;
	}
	//consolidation stress
	for( i=0L; i<4L; i++ )
	{
		ELEM[iElem].sig[i].x = consolid_stress;
		ELEM[iElem].sig[i].y = consolid_stress;
		ELEM[iElem].sig[i].z = consolid_stress;
	}

}//mkQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//B matrix - Q4 element
double mkBQ4( long iE, long iG )
{
	double J[2][2], InvJ[2][2], DetJ;
	double ksi, eta;
	double dN[2][4], b[2][4];
	long row, col, i;
	COOR2 P1, P2, P3, P4, r;
	COOR4 Ni;

	for( row=0; row<4; row++ )
	{
		for( col=0; col<8; col++ )
			B[row][col] = Zero;
	}

	P1 = NODE[ELEM[iE].node[0]];
	P2 = NODE[ELEM[iE].node[1]];
	P3 = NODE[ELEM[iE].node[2]];
	P4 = NODE[ELEM[iE].node[3]];

	/*
	y
	|
	|       P2
	|	  /    \
	|	P1      P3
	|	 \      /
	|	   \  /
	|	    P4
	 ------------------x

	eta
  P2  |  P3
	0---0
	| | |
	|  -------ksi
	|   |
	0---0	
  P1     P4
	*/


	//axisymmetry
	r = GaussQ4( iE, iG );
	Ni = NQ4( iG );
	
	//coordinates of Gauss point
	ksi = GPQ4[iG].ksi;
	eta = GPQ4[iG].eta;

	//derivatives dN/dlocal
	dN[0][0] = NOne * Quarter * ( One - eta );
	dN[0][1] = NOne * Quarter * ( One + eta );
	dN[0][2] = Quarter * ( One + eta );
	dN[0][3] = Quarter * ( One - eta );
	dN[1][0] = NOne * Quarter * ( One - ksi );
	dN[1][1] = Quarter * ( One - ksi );
	dN[1][2] = Quarter * ( One + ksi );
	dN[1][3] = NOne * Quarter * ( One + ksi );

	// Jacobi matrix = dN/dLocal * P(coor)
	J[0][0] = dN[0][0] * P1.x + dN[0][1] * P2.x + dN[0][2] * P3.x + dN[0][3] * P4.x;
	J[0][1] = dN[0][0] * P1.y + dN[0][1] * P2.y + dN[0][2] * P3.y + dN[0][3] * P4.y;
	J[1][0] = dN[1][0] * P1.x + dN[1][1] * P2.x + dN[1][2] * P3.x + dN[1][3] * P4.x;
	J[1][1] = dN[1][0] * P1.y + dN[1][1] * P2.y + dN[1][2] * P3.y + dN[1][3] * P4.y;
			
	//determinant of Jacobi matrix - Jacobian
	DetJ = J[0][0] * J[1][1] - J[0][1] * J[1][0];

	//inverted Jacobi matrix
	InvJ[0][0] = J[1][1] / DetJ;
	InvJ[0][1] = NOne * J[0][1] / DetJ;
	InvJ[1][0] = NOne * J[1][0] / DetJ;
	InvJ[1][1] = J[0][0] / DetJ;

	//B matrix
	for( row=0; row<2; row++ )
	{
		for( col=0; col<4; col++ )
		{
			b[row][col] = Zero;
			for( i=0; i<2; i++ )
				b[row][col] += InvJ[row][i] * dN[i][col];
		}
	}

	B[0][0] = b[0][0]; 
	B[0][2] = b[0][1]; 
	B[0][4] = b[0][2]; 
	B[0][6] = b[0][3];
	B[1][1] = b[1][0]; 
	B[1][3] = b[1][1]; 
	B[1][5] = b[1][2];
	B[1][7] = b[1][3];		
	B[2][0] = B[1][1]; 
	B[2][1] = B[0][0]; 
	B[2][2] = B[1][3]; 
	B[2][3] = B[0][2]; 
	B[2][4] = B[1][5]; 
	B[2][5] = B[0][4]; 
	B[2][6] = B[1][7]; 
	B[2][7] = B[0][6];
	B[3][0] = Ni.one / r.x;
	B[3][2] = Ni.two / r.x;
	B[3][4] = Ni.three / r.x;
	B[3][6] = Ni.four / r.x;

	return DetJ;

}//mkBQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Q4 element - stiffness
void KmQ4( long iElem )
{
	double BTD[8][4], BTDB[8][8], km[8][8], DetJ;
	long row, col, i, j, lrow, lcol, ixy, jxy, iGauss;
	COOR2 r;

	//zeroing km
	for( row=0; row<8; row++ )
	{
		for( col=0; col<8; col++ )
			km[row][col] = Zero;
	}
	
	//for each Gauss point
	for( iGauss=0; iGauss<4; iGauss++ )
	{		
		//axisymmetry
		r = GaussQ4( iElem, iGauss );
		
		STRESS[0] = ELEM[iElem].sig[iGauss].x;
		STRESS[1] = ELEM[iElem].sig[iGauss].y;
		STRESS[2] = ELEM[iElem].sig[iGauss].xy;
		STRESS[3] = ELEM[iElem].sig[iGauss].z;
		
		if( DEPuse == 1L )
			mkDEP( ELEM[iElem].plas[iGauss], STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		else
			mkDEP( 0L, STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars  );
		
		//B matrix
		DetJ = mkBQ4( iElem, iGauss );

		//stiffnes matrix BTDB	
		//first : multiplication BT*D
		//transposed B in multiplication
		for( row=0; row<8; row++ )
		{
			for( col=0; col<4; col++ )
			{
				BTD[row][col] = Zero;
				for( i=0; i<4; i++ ) 
					BTD[row][col] += B[i][row] * DEP[i][col];
			}
		}

		//second : multiplication BTD*B
		for( row=0; row<8; row++ )
		{
			for( col=0; col<8; col++ )
			{
				BTDB[row][col] = Zero;
				for( i=0; i<4; i++ )
					BTDB[row][col] += BTD[row][i] * B[i][col];
			}
		}

		//third : integration DetJ*Wi*BTDB*r, Wi = 1
		for( row=0; row<8; row++ )
		{
			for( col=0; col<8; col++ )
			{
				km[row][col] += DetJ * BTDB[row][col] * r.x;
			}
		}		
	}

	//update global stiffness matrix
	i = 0;
	for( lrow=0; lrow<4; lrow++ )
	{
		for( ixy=0; ixy<2; ixy++ )
		{
			row = 2 * ELEM[iElem].node[lrow] + ixy;
			j = 0;
			for( lcol=0; lcol<4; lcol++ )
			{
				for( jxy=0; jxy<2; jxy++ )
				{
					col = 2 * ELEM[iElem].node[lcol] + jxy;				
					Km[row][col] += km[i][j];
					j+=1;
				}
			}
			i+=1;
		}
	}	
}//KmQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//stresses & strains Q4
void esQ4( long iElem, int opt )
{
	double uQ4[8], Fc, DetJ, Edp_new, sig[4];
	long col, iGauss;
	COOR2 U;
	COOR3 deps;
	
	//retrieving displacement increments
	U = readNODE( u, ELEM[iElem].node[0] );
	uQ4[0] = U.x;
	uQ4[1] = U.y;
	U = readNODE( u, ELEM[iElem].node[1] );
	uQ4[2] = U.x;
	uQ4[3] = U.y;
	U = readNODE( u, ELEM[iElem].node[2] );
	uQ4[4] = U.x;
	uQ4[5] = U.y;
	U = readNODE( u, ELEM[iElem].node[3] );
	uQ4[6] = U.x;
	uQ4[7] = U.y;
   
	//for each Gauss point
	for( iGauss=0L; iGauss<4L; iGauss++ )
	{		
		//B matrix, DetJ (not used here)
		DetJ = mkBQ4( iElem, iGauss );

		//strains increment, de=B*du
		deps = COOR3_ZERO;
		for( col=0L; col<8L; col++ ) 
		{
			deps.x  += B[0][col] * uQ4[col];		
		    deps.y  += B[1][col] * uQ4[col];		
		    deps.xy += B[2][col] * uQ4[col];
		    deps.z  += B[3][col] * uQ4[col];
		}		
		
		if( opt == 0 )
		{
			//stress increment (Newton-Raphson)
			Edp_new = calcDSIG( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].dsig[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		
			//updating incremental stresses & strains
			ELEM[iElem].deps[iGauss].x  += deps.x;
			ELEM[iElem].deps[iGauss].y  += deps.y;
			ELEM[iElem].deps[iGauss].xy += deps.xy;
			ELEM[iElem].deps[iGauss].z  += deps.z;
			ELEM[iElem].dsig[iGauss].x  += dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  += dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy += dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  += dSTRESS[3];
			ELEM[iElem].Edp[iGauss]  = Edp_new;
			//yield value
			sig[0] = ELEM[iElem].sig[iGauss].x + ELEM[iElem].dsig[iGauss].x;
			sig[1] = ELEM[iElem].sig[iGauss].y + ELEM[iElem].dsig[iGauss].y;
			sig[2] = ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy;
			sig[3] = ELEM[iElem].sig[iGauss].z + ELEM[iElem].dsig[iGauss].z;
			Fc = Fmc_MMC( sig, Fi_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ), c_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ));
		}//N-R
		else
		{
			//viscoplastic algorithm

			//minimum DT, dsig=>dSTRESS, dQ=>DQ 
			Fc = calcVP( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].depsvp[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );

			//saving stresses & strains increments
			ELEM[iElem].deps[iGauss] = deps;
			
			ELEM[iElem].dsig[iGauss].x  = dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  = dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy = dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  = dSTRESS[3];
			
			//saving Q-derivatives
			ELEM[iElem].dQ[iGauss].x  = DQ[0];
			ELEM[iElem].dQ[iGauss].y  = DQ[1];
			ELEM[iElem].dQ[iGauss].xy = DQ[2];
			ELEM[iElem].dQ[iGauss].z  = DQ[3];
		}//vp
		
		//updating markers
		ELEM[iElem].Fmc[iGauss] = Fc;
		if( Fc >= Zero ) 
		{
			ELEM[iElem].plas[iGauss] = 1L;
		}
		else
		{
			ELEM[iElem].plas[iGauss] = 0L;
		}

		//pore pressure, u=Du*eps
		//matrix D, U
		mkDU( ELEM[iElem].matpars.Ke );
		ELEM[iElem].u[iGauss] = initial_u;
		ELEM[iElem].u[iGauss] += ELEM[iElem].matpars.Ke * ( ELEM[iElem].eps[iGauss].x + ELEM[iElem].eps[iGauss].y + ELEM[iElem].eps[iGauss].z );
		
		//Gauss point
		ELEM[iElem].Gauss[iGauss] = GaussQ4( iElem, iGauss );		
	}//Gauss point
}//esQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//viscoplastic strain & stress increment
void vpQ4( long iElem )
{
	double ddepsvp[4], dsigvp[4];
	long row, col, iGauss;

	mkDE( ELEM[iElem].matpars.E, ELEM[iElem].matpars.ni, ELEM[iElem].matpars.Ke );
	for( iGauss=0L; iGauss<4L; iGauss++ )
	{
		if( ELEM[iElem].Fmc[iGauss] >= Zero )
		{
			//viscoplastic strain increment
			ddepsvp[0] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].x;
			ddepsvp[1] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].y;
			ddepsvp[2] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].xy;
			ddepsvp[3] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].z;
	
			//plastic deviatoric strain
			ELEM[iElem].Edp[iGauss] += DT * ELEM[iElem].Fmc[iGauss];

			//updating viscoplastic strain increment
			ELEM[iElem].depsvp[iGauss].x  += ddepsvp[0];
			ELEM[iElem].depsvp[iGauss].y  += ddepsvp[1];
			ELEM[iElem].depsvp[iGauss].xy += ddepsvp[2];
			ELEM[iElem].depsvp[iGauss].z  += ddepsvp[3];

			//viscoplastic stress increment
			for( row=0L; row<4L; row++ )
			{
				dsigvp[row] = Zero;
				for( col=0L; col<4L; col++ )
				{
					dsigvp[row] += DE[row][col] * ddepsvp[col];
				}
			}
		}//F>0
		else
		{
			for( row=0L; row<4L; row++ ) dsigvp[row] = Zero;
		}
			
		//saving viscoplastic stress increment
		ELEM[iElem].dsigvp[iGauss].x  = dsigvp[0];
		ELEM[iElem].dsigvp[iGauss].y  = dsigvp[1];
		ELEM[iElem].dsigvp[iGauss].xy = dsigvp[2];
		ELEM[iElem].dsigvp[iGauss].z  = dsigvp[3];
	}//iGauss
}//vpQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//body loads, new version as residual loads
void rlQ4( long iElem, int opt )
{
	int iGauss, row;
	double DetJ, BTS[8], fR[8];
	COOR2 r;

	//zeroing
	for( row=0; row<8; row++ ) fR[row] = Zero;
	
	//integration
	for( iGauss=0; iGauss<4; iGauss++ )
	{
		//B matrix
		DetJ = mkBQ4( iElem, iGauss );

		if( opt==0 )
		{
			//multiplication BT*(sig+dsig)
			for( row=0; row<8; row++ )
			{
				BTS[row]  = B[0][row] * ( ELEM[iElem].sig[iGauss].x  + ELEM[iElem].dsig[iGauss].x );
				BTS[row] += B[1][row] * ( ELEM[iElem].sig[iGauss].y  + ELEM[iElem].dsig[iGauss].y );
				BTS[row] += B[2][row] * ( ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy );
				BTS[row] += B[3][row] * ( ELEM[iElem].sig[iGauss].z  + ELEM[iElem].dsig[iGauss].z );
			}
		}
		else
		{
			//multiplication BT*dsigvp
			for( row=0; row<8; row++ )
			{
				BTS[row]  = B[0][row] * ELEM[iElem].dsigvp[iGauss].x;
				BTS[row] += B[1][row] * ELEM[iElem].dsigvp[iGauss].y;
				BTS[row] += B[2][row] * ELEM[iElem].dsigvp[iGauss].xy;
				BTS[row] += B[3][row] * ELEM[iElem].dsigvp[iGauss].z;
			}
		}
		
		//axisymmetry
		r = GaussQ4( iElem, iGauss );
		
		//integration
		for( row=0; row<8; row++ )
		{
			fR[row] += DetJ * GPQ4[iGauss].w * BTS[row] * r.x;
		}
	}
	
	updatefR( ELEM[iElem].node[0], fR[0], fR[1] );
	updatefR( ELEM[iElem].node[1], fR[2], fR[3] );
	updatefR( ELEM[iElem].node[2], fR[4], fR[5] );
	updatefR( ELEM[iElem].node[3], fR[6], fR[7] );

}//rlQ4
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double N4( long iElem, int iNode )
{
	int iGauss;
	double vN, NQ, DetJ;
	COOR2 r;
	
	vN = Zero;
	for(iGauss=0L; iGauss<4L; iGauss++)
	{
		switch(iNode)
		{
			case 1: NQ = NQ4(iGauss).one;
					break;
			case 2: NQ = NQ4(iGauss).two;
					break;
			case 3:	NQ = NQ4(iGauss).three;
					break;
			case 4:	NQ = NQ4(iGauss).four;
					break;
		}
		//axisymmetry
		r = GaussQ4( iElem, iGauss );
		DetJ = mkBQ4( iElem, iGauss );
		vN += DetJ * GPQ4[iGauss].w * NQ * r.x;
	}
	return vN;
}//N4
//-------------------------------------------------------------------------------------

/* END Q4 */

#endif
