/*
************************
***** Main program *****
*** P.E.Srokosz, 2006 **
************************
*/


/***********************
        HEADERS
************************/

//system files
#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>

//FEM library files
#include "FEMLibAX.h"
#include "FEMVars.h"
#include "TQAX.H"
#include "MCPLAX.H"
#include "MMC.H"
#include "IOToolsAX.h"

//FEM elements
#include "T3AX.H"
#include "T6AX.H"
#include "Q4AX.H"
#include "Q8AX.H"
#include "Q9AX.H"


/***********************
	   VARIABLES
************************/
	
static char BMPname[100], INPname[100], MSHname[100], OUTname[100], EPSname[100], SIGname[100];
static char CTRname[100], INCname[100], slincr[33], workstr[33], K0, LOADname[100];
static char GIDname[100], DISPname[100], SURFname[100], CalcMonitor = 0, TRIAXname[100];
static long i, j, ichange, exitcode, done, lincr, total_incr, total_iter, break_calc;
static double ctrl, ctrl_prev = (double)0.0, incr, normf_ext, du_elas[MaxDim];
static double u_prev[MaxDim], f_incr[MaxDim], df[MaxDim];
static double f_constr[MaxDim];


/*************************
 EXTRACTED MAIN PROCEDURE
**************************/

int CalculateIt( void )
{
	FILE *stream;
	COOR2L pv;
	char incrchange = (char)0;

	//opening stream for status and warning messages
	if( CalcMonitor == 1 ) stream = fopen( "CalcMonitor.txt", "wt");
		
	incr = Zero;
	total_iter = 0L;
	total_incr = 0L;
	break_calc = 0L;
	
	/* Info
			f_total : total load vector (from input)
			df		: load increment
			f       : working vectors
	*/
	
	if( VPuse == 1L )
	{
		/***** VISCOPLASTIC *****/
		/* Info
				f_constr: constraints in load vector
				f_incr	: total increment load vector with residuals
		*/
				
		if( CalcMonitor == 1 ) fprintf( stream, "Viscoplastic algorithm.\n");
		printf( "\nViscoplastic algorithm.\n");

		//Stiffness matrix
		if( CalcMonitor == 1 ) fprintf( stream, "Forming stiffness matrix: ");
		printf("Forming stiffness matrix: ");
		createKm( );	
		printf("OK.\n");
		if( CalcMonitor == 1 ) fprintf( stream, "OK.\n");
		
		//Boundary conditions
		fZERO( );
		Constraints( );

		//Saving constraints in load vector
		for( i=0L; i<NDIM; i++ ) f_constr[i] = f[i];

		//Inverting stiffness matrix
		printf("Inverting stiffness matrix: ");
		if( CalcMonitor == 1 ) fprintf( stream, "Inverting stiffness matrix: ");
		//Save_par("Parameters.txt");
		//Save_Km("Km.txt");
		exitcode = KmINVERSE( );
		if( exitcode != 0 )
		{
			if( CalcMonitor == 1 ) fprintf( stream, "Error!\nKmINVERSE exitcode : \n", exitcode );
			printf( "Error!\nKmINVERSE exitcode : \n", exitcode );
			fclose( stream );
			return exitcode;
		}
		printf("OK.\n");
		if( CalcMonitor == 1 ) fprintf( stream, "OK.\n");					

		//load loop
		do
		{
			incr += delta;
			lincr += 1L;

			//Start
			INCREMENT = incr / INClimit;
			printf("\nLoad : %.2lf%% \n", INCREMENT * (double) 100.0);
			if( CalcMonitor == 1 ) fprintf( stream, "\nLoad : %.2lf%%\n", INCREMENT * (double) 100.0);
			
			//Zeroing elements increments
			ZeroELEMincr( );

			//First iteration
			Iter = 1L;

			//1.A.Total load vector without constrains
			//1.B.Building calculation load vector (f_constr + (df - constr))
			for( i=0L; i<NDIM; i++ ) fResidual[i] = df[i];
			fZERO( );
			upfTQ( );
			for( i=0L; i<NDIM; i++ ) 
			{
				f_ext[i] += f[i];
				f[i] += f_constr[i];
			}
			
			//2.Calculating unknowns
			for( i=0L; i<NDIM; i++ )
			{
				u[i] = Zero;
				for( j=0L; j<NDIM; j++ )
				{
					u[i] += Km[i][j] * f[j];
				}
			}
			
			//3.Reallocation of results, cleaning reactions, update prescribed displacements
			ufTQ( df, 1 );			
			if( _finite( normV( u, NDIM )) == 0 ) 
			{
				break_calc = 1L;
				break;
			}
		
			//4.Strains & stresses, time step
			//4.A:Elasticity
			DT = One;
			calcEpsSig( VPuse );
			//4.B:Viscoplasticity
			calcEpsSigVP( );
			
			//5.Body loads consistent to viscoplastic stress increments
			ctrl = ResidualLoads( VPuse );

			//6.Saving displacement vector
			for( i=0L; i<NDIM; i++ ) 
			{
				u_prev[i] = u[i];
				//elastic part of displacements (first increment HAVE TO BE ELASTIC)
				if( lincr == 1L ) du_elas[i] = u[i];
				u_elas[i] += du_elas[i];
			}
			
			//7.Updating calculation load vector ( f + (fResiduals - constr))
			upfTQ( );
			
			//8.Calculates number of plastic & violated Gauss points
			pv = Report( );
			printf( "I/P/V/T/dt : %d/%d/%d/%d/%E\n", Iter, pv.x, pv.y, totalGP, DT );
			if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/dt : %d/%d/%d/%d/%E\n", Iter, pv.x, pv.y, totalGP, DT );
			
			//9.Iterations, if yield violation exists in system
			if( pv.y > 0L ) 
			{
				//Nothing done yet
				done = 1L;
				do
				{						
					//Next iteration
					Iter += 1L;
				
					//1.I.Saving increment load vector, updating constraints
					for( i=0L; i<NDIM; i++ ) 
					{
						f_incr[i] = f[i];
						f[i] += f_constr[i];
					}
					
					//2.I.Recalculating displacement vector					
					for( i=0L; i<NDIM; i++ )
					{
						u[i] = Zero;
						for( j=0L; j<NDIM; j++ )
						{
							u[i] += Km[i][j] * f[j];
						}
					}
					//errors
					if( _finite( normV( u, NDIM )) == 0 )
					{
						break_calc = 1L;
						Save_all( "Errors.txt");
						break;
					}

					//3.I.Reallocation of results, cleaning reactions
					ufTQ( f_incr, 1 );

					//4.I.Strains & stresses
					//4.A:Elasticity
					DT = One;
					calcEpsSig( VPuse );
					//4.B:Viscoplasticity
					calcEpsSigVP( );
			
					//5.I.Body loads consistent to viscoplastic stress increments
					ctrl = ResidualLoads( VPuse );
					
					//6.I.Calculating difference vector
					for( i=0L; i<NDIM; i++ ) u_prev[i] -= u[i];
					
					//7.I.Updating calculation load vector (f + (fResiduals - constr))
					upfTQ( );
					
					//8.I.Vectors norms, saving actual displacement vector
					ctrl = normV( u_prev, NDIM);
					NRctrl = ctrl / normV( u, NDIM );
					for( i=0L; i<NDIM; i++ ) u_prev[i] = u[i];

					//9.I.Convergence criteria
					if( NRctrl < NRtol ) done = 0L;
					if( fabs( NRctrl - ctrl_prev ) < (double) 0.001 * NRtol ) done = 0L;
					ctrl_prev = NRctrl;

					//10.I.Calculates number of plastic & violated Gauss points
					pv = Report( );
					printf( "I/P/V/T/dt/E : %d/%d/%d/%d/%E/%E%%\n", Iter, pv.x, pv.y, totalGP, DT, NRctrl * (double) 100.0 );
					if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/dt/E/norm(du) : %d/%d/%d/%d/%E/%E/%E\n", Iter, pv.x, pv.y, totalGP, DT, NRctrl, ctrl );
					if( pv.y > Vcrit * pv.x ) done = 1L;
					//auto increment
					if( INCauto > 0L )
					{
						if( Iter > ( 0.85 * NRmax ))
						{
							if( incrchange<INCauto)
							{
								delta *= Half;
								for( i=0L; i<NDIM; i++ ) df[i] *= Half;
								incrchange++;
								printf( "Changing step size: %d\n", incrchange );
								if( CalcMonitor == 1 ) fprintf( stream, "Changing step size: %d\n", incrchange );
							}
						}
					}
					if( Iter > NRmax ) done = 0L;
				}
				while( done > 0L );
			}

			total_iter += Iter;

			printf("\n");
			if( CalcMonitor == 1 ) fprintf( stream, "\n" );
		
			//10.Updating total strains & stresses in elements
			upEpsSig( );

			//11.Updating total & incremental displacements
			if( break_calc == 0L )
			{
				for( i=0L; i<NDIM; i++ )
				{
					u_act[i] += u[i];
					u_incr[i] = u[i];
				}
			}
			else
			{
				for( i=0L; i<NDIM; i++ )
				{
					u_act[i] += u_prev[i];
					u_incr[i] = u_prev[i];
				}
			}
			
			//12.Saving results to disk
			strcpy( INCname, OUTname );
			strcpy( workstr, "." );
			if( lincr <100L ) strcat( workstr, "0" );
			if( lincr <10L ) strcat( workstr, "0" );
			ltoa( lincr, slincr, 10 );
			strcat( workstr, slincr );
			strcat( INCname, workstr );
			strcat( INCname, ".OUT.DAT" );
			Save_all( INCname );
			Save4GID7( GIDname, lincr );
			SaveTriax( TRIAXname, 1L );
			
			//saving displacements and reactions on selected nodes
			for( i=0L; i<NDIM; i++ ) fR_copy[i] = fResidual[i];
			fRZERO( );
			ctrl = ResidualLoads( 0 );
			Save_monitor( OUTname, lincr );
			for( i=0L; i<NDIM; i++ ) fResidual[i] = fR_copy[i];

			CreateSign( CTRname, Iter );

			//13.Break, if maximum violated GP exceeded
			if( pv.y > Vcrit * pv.x ) 
			{
				printf( "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
				if( CalcMonitor == 1 ) fprintf( stream, "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
				if( CalcMonitor == 1 ) fprintf( stream, "Exit.\n" );
				incr = INCmax;
			}	

			//14.Break, if tolerated error exceeded (collapse)
			if( NRctrl > Collapse ) 
			{
				printf( "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
				if( CalcMonitor == 1 ) fprintf( stream, "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
				if( CalcMonitor == 1 ) fprintf( stream, "Exit.\n" );
				incr = INCmax;
			}
			
			//15.Break, if numerical instability occurs
			if( break_calc > 0L )
			{
				printf( "Error! Numerical instability.\n" );
				if( CalcMonitor == 1 ) fprintf( stream, "Error! Numerical instability.\n" );
				if( CalcMonitor == 1 ) fprintf( stream, "Exit.\n" );
				incr = INCmax;
			}
		}
		while( incr < ( INCmax - delta / Six ));
		total_incr = lincr;
	}//viscoplastic
	else
	{
		/***** NEWTON-RAPHSON *****/

		if ( DEPuse == 1L )
		{
			/***** TANGENT STIFFNESS *****/
		
			printf( "\nNewton-Raphson algorithm. Tangent stiffness.\n");
			if( CalcMonitor == 1 ) fprintf( stream, "Newton-Raphson algorithm. Tangent stiffness.\n");
			ichange = 0L;
			//load loop
			do
			{
				//subiterations controls
				IterCtrl = 0L;
				RetCtrl = 0L;

				incr += delta;
				lincr += 1L;
		
				//Start
				INCREMENT = incr / INClimit;
				printf("Load : %.2lf%% \n", INCREMENT * (double) 100.0);
				if( CalcMonitor == 1 ) fprintf( stream, "Load : %.2lf%%\n", INCREMENT * (double) 100.0);
			
				//Zeroing strain & stress increments
				ZeroELEMincr( );
		
				//Total load vector without constrains
				for( i=0L; i<NDIM; i++ ) fResidual[i] = df[i];
				fZERO( );
				upfTQ( );
				for( i=0L; i<NDIM; i++ ) f_ext[i] += f[i];
				normf_ext = normV( f_ext, NDIM );

				//First iteration
				Iter = 1L;
		
				//1.Stiffness matrix
				createKm( );	
		
				//2.Boundary conditions, reallocation of unknowns
				Constraints( );
		
				//3.Solving system of equations
				printf("Gauss elimination process: ");
				if( CalcMonitor == 1 ) fprintf( stream, "Gauss elimination process: ");
				exitcode = GAUSS( );
				if( exitcode != 0 )
				{
					if( CalcMonitor == 1 ) fprintf( stream, "Error!\n Iter: %d. GAUSS exitcode : \n", Iter, exitcode );
					printf( "Error!\n Iter: %d. GAUSS exitcode : \n", Iter, exitcode );
					fclose( stream );
					return exitcode;
				}

				printf("OK. ");
				if( CalcMonitor == 1 ) fprintf( stream, "OK. ");
			
				//4.Reallocation of results, cleaning reactions
				ufTQ( f_ext, 1 );
				for( i=0L; i< NDIM; i++ )
				{
					//elastic part of displacements (first increment HAVE TO BE ELASTIC)
					if( lincr == 1L ) du_elas[i] = u[i];
				}
				if( break_calc < lincr ) 
				{
					for( i=0L; i< NDIM; i++ ) u_elas[i] += du_elas[i];
					break_calc = lincr;
				}
				else
				{
					for( i=0L; i< NDIM; i++ ) u_elas[i] -= du_elas[i];
				}

				//5.Elasto-plastic strains & stresses
				calcEpsSig( VPuse );
			
				//6.Internal forces consistent to total stresses
				ctrl = ResidualLoads( VPuse );
				for( i=0L; i<NDIM; i++ ) fResidual[i] *= NOne;

				//7.Residual load vector ( f - (fResidual - constr))
				upfTQ( );

				//8.Updating incremental displacements
				for( i=0L; i<NDIM; i++ ) u_incr[i] = u[i];
			
				//9A.Convergence criteria				
				NRctrl = normV( f, NDIM ) / normf_ext; 
				ctrl_prev = NRctrl;

				//9B. Overrelaxation
				for( i=0L; i<NDIM; i++ ) f[i] *= NRRelax;
		
				//10.Calculates number of plastic & violated Gauss points
				pv = Report( );
				printf( "I/P/V/T/E/C1/C2 : %d/%d/%d/%d/%E%%/%d/%d\n", Iter, pv.x, pv.y, totalGP, NRctrl * (double) 100.0, IterCtrl, RetCtrl );
				if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/E/norm(R)/C1/C2 : %d/%d/%d/%d/%E/%E/%d/%d\n", Iter, pv.x, pv.y, NELEM * 4L, NRctrl, ctrl, IterCtrl, RetCtrl );

				//11.Iterations, when unbalanced forces exist in the system
				if( NRctrl > NRtol ) 
				{
					//Nothing done yet
					done = 1L;
					do
					{						
						//subiteration controls
						IterCtrl = 0L;
						RetCtrl = 0L;
						
						//Next iteration
						Iter += 1L;

						//1.I.Stiffness matrix
						createKm( );	
			
						//2.I.Boundary conditions, reallocation of unknowns
						Constraints( );
				
						//3.I.Solving system of equations
						printf("Gauss elimination process: ");
						if( CalcMonitor == 1 ) fprintf( stream, "Gauss elimination process: ");
						exitcode = GAUSS( );
						if( exitcode != 0 )
						{
							if( CalcMonitor == 1 ) fprintf( stream, "Error!\n Iter: %d. GAUSS exitcode : \n", Iter, exitcode );
							printf( "Error!\n Iter: %d. GAUSS exitcode : \n", Iter, exitcode );
							fclose( stream );
							return exitcode;
						}
						printf("OK. ");
						if( CalcMonitor == 1 ) fprintf( stream, "OK. ");

						//4.I.Reallocation of results, cleaning reactions
						ufTQ( f_ext, 1 );
				
						//5.I.Elasto-plastic strains & stresses
						calcEpsSig( VPuse );
			
						//6.I.Internal forces consistent to stresses
						ctrl = ResidualLoads( VPuse );
						for( i=0L; i<NDIM; i++ ) fResidual[i] *= NOne;

						//7.Residual load vector ( f - (fResidual - constr))
						upfTQ( );
						
						//8.I.Updating incremental displacements
						for( i=0L; i<NDIM; i++ ) u_incr[i] += u[i];
			
						//9A.I.Convergence criteria
						NRctrl = normV( f, NDIM ) / normf_ext; 
						if( NRctrl < NRtol ) done = 0L;
						if( fabs( NRctrl - ctrl_prev ) < (double) 0.001 * NRtol ) done = 0L;
						ctrl_prev = NRctrl;
						if( Iter > NRmax ) done = 0L;
						
						//9B.I.Overrelaxation
						for( i=0L; i<NDIM; i++ ) f[i] *= NRRelax;

						//10.I.Calculates number of plastic & violated Gauss points
						pv = Report( );
						printf( "I/P/V/T/E/C1/C2 : %d/%d/%d/%d/%E%%/%d/%d\n", Iter, pv.x, pv.y, totalGP, NRctrl * (double) 100.0, IterCtrl, RetCtrl );
						if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/E/norm(R)/C1/C2 : %d/%d/%d/%d/%E/%E/%d/%d\n", Iter, pv.x, pv.y, NELEM * 4L, NRctrl, ctrl, IterCtrl, RetCtrl );
						if(( NRctrl > ( Four * Collapse )) and ( Iter > 10L )) done = 0L;
					}
					while( done > 0L );
				}

				total_iter += Iter;
				
				printf("\n");
				if( CalcMonitor == 1 ) fprintf( stream, "\n" );
		
				//12.A.Flag : Maximum violated GP exceeded
				done = 0L;
				if( pv.y > Vcrit * pv.x ) 
				{
					printf( "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
					if( CalcMonitor == 1 ) fprintf( stream, "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
					done = 1L;
				}	
				
				//12.B.Tolerated error exceeded (collapse)
				if( NRctrl > Collapse ) 
				{
					printf( "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
					if( CalcMonitor == 1 ) fprintf( stream, "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
					done = 1L;
				}

				//13.Change step size if INCauto checked
				if( done > 0L )
				{
					if( INCauto > 0L )
					{					
						if( ichange < INCauto )
						{
							//decreasing increment size
							ichange += 1L;
							incr -= delta;
							lincr -= 1L;
							delta *= Half;
							for( i=0L; i<NDIM; i++ ) fResidual[i] = df[i];
							fZERO( );
							upfTQ( );
							for( i=0L; i<NDIM; i++ ) 
							{
								f_ext[i] -= f[i];
								df[i] *= Half;
								du_elas[i] *= Half;
							}
							printf( "%d : Decreasing step size.\n", ichange );
							if( CalcMonitor == 1 ) fprintf( stream, "%d : Decreasing step size.\n", ichange );
							done = 1L;
							Iter = 0L;
						}
						else
						{
							if( CalcMonitor == 1 ) fprintf( stream, "Exit.\n" );
							incr = INCmax;
							done = 0L;
						}
					}
					else
						done = 0L;
				}

				//14.Saving results
				if( done == 0L )
				{
					//updating elements
					upEpsSig( );
					//updating total loads and displacements
					for( i=0L; i<NDIM; i++ ) u_act[i] += u_incr[i];
					//saving results
					strcpy( INCname, OUTname );
					strcpy( workstr, "." );
					if( lincr <100L ) strcat( workstr, "0" );
					if( lincr <10L ) strcat( workstr, "0" );
					ltoa( lincr, slincr, 10 );
					strcat( workstr, slincr );
					strcat( INCname, workstr );
					strcat( INCname, ".OUT.DAT" );
					Save_all( INCname );
					Save4GID7( GIDname, lincr );
					Save_monitor( OUTname, lincr );
					CreateSign( CTRname, Iter );
				}			
			}
			while( incr < ( INCmax - delta / Six ));
			total_incr = lincr;
		}//DEPuse==1
		else
		{
			/***** CONSTANT STIFFNESS *****/
					
			printf( "\nNewton-Raphson algorithm. Constant stiffness.\n");
			if( CalcMonitor == 1 ) fprintf( stream, "Newton-Raphson algorithm. Constant stiffness.\n");
		
			//Stiffness matrix
			if( CalcMonitor == 1 ) fprintf( stream, "Forming stiffness matrix: ");
			printf("Forming stiffness matrix: ");
			createKm( );	
			printf("OK.\n");
			if( CalcMonitor == 1 ) fprintf( stream, "OK.\n");
		
			//Boundary conditions
			fZERO( );
			Constraints( );

			//Saving constraints in load vector
			for( i=0L; i<NDIM; i++ ) f_constr[i] = f[i];

			//Inverting stiffness matrix
			printf("Inverting stiffness matrix: ");
			if( CalcMonitor == 1 ) fprintf( stream, "Inverting stiffness matrix: ");
			exitcode = KmINVERSE( );
			if( exitcode != 0 )
			{
				if( CalcMonitor == 1 ) fprintf( stream, "Error!\n Iter: %d. KmINVERSE exitcode : \n", Iter, exitcode );
				printf( "Error!\n Iter: %d. KmINVERSE exitcode : \n", Iter, exitcode );
				fclose( stream );
				return exitcode;
			}
			printf("OK.\n");
			if( CalcMonitor == 1 ) fprintf( stream, "OK.\n");					
			ichange = 0L;
			//load loop
			do
			{
				//subiterations controls
				IterCtrl = 0L;
				RetCtrl = 0L;

				incr += delta;
				lincr += 1L;
		
				//Start
				INCREMENT = incr / INClimit;
				printf("Load : %.2lf%% \n", INCREMENT * (double) 100.0);
				if( CalcMonitor == 1 ) fprintf( stream, "Load : %.2lf%%\n", INCREMENT * (double) 100.0);
			
				//Zeroing strain & stress increments
				ZeroELEMincr( );
		
				//First iteration
				Iter = 1L;

				//1.A.Total load vector without constrains
				//1.B.Building calculation load vector (f_constr + (df - constr))
				for( i=0L; i<NDIM; i++ ) fResidual[i] = df[i];
				fZERO( );
				upfTQ( );
				for( i=0L; i<NDIM; i++ )
				{
					f_ext[i] += f[i];
					f[i] += f_constr[i];
				}
				normf_ext = normV( f_ext, NDIM );
			
				//2.Calculating unknowns
				for( i=0L; i<NDIM; i++ )
				{
					u[i] = Zero;
					for( j=0L; j<NDIM; j++ )
					{
						u[i] += Km[i][j] * f[j];
					}
				}
							
				//3.Reallocation of results, cleaning reactions
				ufTQ( f_ext, 1 );
				for( i=0L; i< NDIM; i++ )
				{
					//elastic part of displacements (first increment HAVE TO BE ELASTIC)
					if( lincr == 1L ) du_elas[i] = u[i];
				}
				if( break_calc < lincr ) 
				{
					for( i=0L; i< NDIM; i++ ) u_elas[i] += du_elas[i];
					break_calc = lincr;
				}
				else
				{
					for( i=0L; i< NDIM; i++ ) u_elas[i] -= du_elas[i];
				}
		
				//4.Elasto-plastic strains & stresses
				calcEpsSig( VPuse );
			
				//5.Internal forces consistent to total stresses
				ctrl = ResidualLoads( VPuse );
				for( i=0L; i<NDIM; i++ ) fResidual[i] *= NOne;

				//6.Residual load vector ( f - (fResidual - constr))
				upfTQ( );

				//7.Updating incremental displacements
				for( i=0L; i<NDIM; i++ ) u_incr[i] = u[i];
			
				//8A.Convergence criteria				
				NRctrl = normV( f, NDIM ) / normf_ext; 
				ctrl_prev = NRctrl;

				//8B.Overrelaxation
				for( i=0L; i<NDIM; i++ ) f[i] *= NRRelax;

				//9.Calculates number of plastic & violated Gauss points
				pv = Report( );
				printf( "I/P/V/T/E/C1/C2 : %d/%d/%d/%d/%E%%/%d/%d\n", Iter, pv.x, pv.y, totalGP, NRctrl * (double) 100.0, IterCtrl, RetCtrl );
				if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/E/norm(R)/C1/C2 : %d/%d/%d/%d/%E/%E/%d/%d\n", Iter, pv.x, pv.y, NELEM * 4L, NRctrl, ctrl, IterCtrl, RetCtrl );

				//10.Iterations, when unbalanced forces exist in the system
				if( NRctrl > NRtol ) 
				{
					//Nothing done yet
					done = 1L;
					do
					{						
						//subiterations controls
						IterCtrl = 0L;
						RetCtrl = 0L;
						
						//Next iteration
						Iter += 1L;

						//1.I.Updating constraints
						for( i=0; i<NDIM; i++ ) f[i] += f_constr[i];
					
						//2.I.Recalculating displacement vector					
						for( i=0L; i<NDIM; i++ )
						{
							u[i] = Zero;
							for( j=0L; j<NDIM; j++ )
							{
								u[i] += Km[i][j] * f[j];
							}
						}
						
						//3.I.Reallocation of results, cleaning reactions
						ufTQ( f_ext, 1 );
				
						//4.I.Elasto-plastic strains & stresses
						calcEpsSig( VPuse );
			
						//5.I.Internal forces consistent to stresses
						ctrl = ResidualLoads( VPuse );
						for( i=0L; i<NDIM; i++ ) fResidual[i] *= NOne;

						//6.Residual load vector ( f - (fResidual - constr))
						upfTQ( );
						
						//7.I.Residual loads, updating variables
						for( i=0L; i<NDIM; i++ ) u_incr[i] += u[i];
			
						//8A.I.Convergence criteria
						NRctrl = normV( f, NDIM ) / normf_ext; 
						if( NRctrl < NRtol ) done = 0L;
						if( fabs( NRctrl - ctrl_prev ) < (double) 0.001 * NRtol ) done = 0L;
						ctrl_prev = NRctrl;
						if( Iter > NRmax ) done = 0L;
						
						//8B.I.Overrelaxation
						for( i=0L; i<NDIM; i++ ) f[i] *= NRRelax;

						//9.I.Calculates number of plastic & violated Gauss points
						pv = Report( );
						printf( "I/P/V/T/E/C1/C2 : %d/%d/%d/%d/%E%%/%d/%d\n", Iter, pv.x, pv.y, totalGP, NRctrl * (double) 100.0, IterCtrl, RetCtrl );
						if( CalcMonitor == 1 ) fprintf( stream, "I/P/V/T/E/norm(R)/C1/C2 : %d/%d/%d/%d/%E/%E/%d/%d\n", Iter, pv.x, pv.y, NELEM * 4L, NRctrl, ctrl, IterCtrl, RetCtrl );
						if(( NRctrl > ( Four * Collapse )) and ( Iter > 10L )) done = 0L;
					}
					while( done > 0L );
				}

				total_iter += Iter;
				
				printf("\n");
				if( CalcMonitor == 1 ) fprintf( stream, "\n" );
		
				//11.A.Flag : Maximum violated GP exceeded
				done = 0L;
				if( pv.y > Vcrit * pv.x ) 
				{
					printf( "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
					if( CalcMonitor == 1 ) fprintf( stream, "Maximum number of violated points (%.2lf%%) exceeded %.2lf%%\n", (double) pv.y / (double) pv.x * (double) 100.0, Vcrit * (double) 100.0 );
					done = 1L;
				}	
				
				//11.B.Tolerated error exceeded (collapse)
				if( NRctrl > Collapse ) 
				{
					printf( "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
					if( CalcMonitor == 1 ) fprintf( stream, "System collapsed! Error (%.2lf%%) exceeded %.2lf%%\n", NRctrl * (double) 100.0, Collapse * (double) 100.0 );
					done = 1L;
				}

				//12.Change step size if INCauto checked
				if( done > 0L )
				{
					if( INCauto > 0L )
					{					
						if( ichange < INCauto )
						{
							//decreasing increment size
							ichange += 1L;
							incr -= delta;
							lincr -= 1L;
							delta *= Half;
							for( i=0L; i<NDIM; i++ ) fResidual[i] = df[i];
							fZERO( );
							upfTQ( );
							for( i=0L; i<NDIM; i++ ) 
							{
								f_ext[i] -= f[i];
								df[i] *= Half;
								du_elas[i] *= Half;
							}
							printf( "%d : Decreasing step size.\n", ichange );
							if( CalcMonitor == 1 ) fprintf( stream, "%d : Decreasing step size.\n", ichange );
							done = 1L;
							Iter = 0L;
						}
						else
						{
							if( CalcMonitor == 1 ) fprintf( stream, "Exit.\n" );
							incr = INCmax;
							done = 0L;
						}
					}
					else
						done = 0L;
				}

				//13.Saving results
				if( done == 0L )
				{
					//updating elements
					upEpsSig( );
					//updating total loads and displacements
					for( i=0L; i<NDIM; i++ ) u_act[i] += u_incr[i];
					//saving results
					strcpy( INCname, OUTname );
					strcpy( workstr, "." );
					if( lincr <100L ) strcat( workstr, "0" );
					if( lincr <10L ) strcat( workstr, "0" );
					ltoa( lincr, slincr, 10 );
					strcat( workstr, slincr );
					strcat( INCname, workstr );
					strcat( INCname, ".OUT.DAT" );
					Save_all( INCname );
					Save4GID7( GIDname, lincr );
					Save_monitor( OUTname, lincr );
					CreateSign( CTRname, Iter );
				}			
			}
			while( incr < ( INCmax - delta / Six ));
			total_incr = lincr;
		}//DEPuse==0
	}//Newton-Raphson

	/***********************
             FINAL
	************************/
	
	//Internal forces consistent to stresses
	ctrl = ResidualLoads( 0 );
	fZERO( );
	upfTQ( );
	for( i=0L; i<NDIM; i++ ) f[i] -= f_ext[i];
	ctrl = normV( f, NDIM ) / normV( f_ext, NDIM );
	if( CalcMonitor == 1 ) Save_vec("f_int.txt", fResidual );
	if( CalcMonitor == 1 ) Save_vec("f_ext.txt", f_ext );
	if( CalcMonitor == 1 ) Save_vec("f_tot.txt", f_total );
	if( CalcMonitor == 1 ) Save_vec("f_res.txt", f );
	Save4SurferMatlab( SURFname );

	printf("\n");
	printf( "Total error: %E%%\n", ctrl * (double) 100.0 );
	if( CalcMonitor == 1 ) fprintf( stream, "\n");
	if( CalcMonitor == 1 ) fprintf( stream, "Total error: %E%%\n", ctrl * (double) 100.0 );
	if( CalcMonitor == 1 ) fprintf( stream, "Total increments: %d\n", total_incr );
	if( CalcMonitor == 1 ) fprintf( stream, "Total iterations: %d\n", total_iter );
	printf( "Total increments: %d\n", total_incr );
	printf( "Total iterations: %d\n", total_iter );

	//closing monitor stream 	
	if( CalcMonitor == 1 ) fclose(stream);
	return 0;
}//CalculateIt

/***********************
      MAIN PROGRAM
************************/

int main(int argc, char* argv[])
{

	double dincr;

	//64 bits by default (in linker list : FP10.obj) 
	_controlfp(_PC_64, _MCW_PC);

	/***********************
          READING DATA
	************************/
	
	//without saving comments on file
	CalcMonitor = 0;

	printf( "Finite Element Analysis, P.E.Srokosz, 2006 (c)\n" );
	printf( "Axisymmetry implemented, BETA version 2009 (c)\n" );

	//program needs parameter - name of input file
	if( argc < 2 )
	{
		printf( "Syntax : program.exe input_file [K0]\n" );
		printf( "K0 : reads stress state calculated previously.\n" );
		return 0;
	}

	INITALL( );

	printf( "\nInput data file : %s \n",argv[1] );
	printf( "\nGeostatic stress calculated: " );
	if( argc == 3 ) 
	{
		printf( "YES \n\n");
		K0 = 1;
	}
	else
	{
		printf( "NO \n\n");
		K0 = 0;
	}

	
	//working file names
	strcpy( INPname, argv[1] );
	strcat( INPname, ".INP" );
	strcpy( MSHname, argv[1] );
	strcat( MSHname, ".MSH.INP" );
	strcpy( BMPname, argv[1] );
	strcat( BMPname, ".BMP" );
	strcpy( OUTname, argv[1] );
	strcpy( EPSname, argv[1] );
	strcat( EPSname, ".EPS.TXT" );
	strcpy( SIGname, argv[1] );
	strcat( SIGname, ".SIG.TXT" );
	strcpy( CTRname, argv[1] );
	strcat( CTRname, ".CTR" );
	strcpy( LOADname, argv[1] );
	strcat( LOADname, ".LOAD.INP" );
	strcpy( DISPname, argv[1] );
	strcat( DISPname, ".DISP.INP" );
	strcpy( GIDname, argv[1] );
	strcat( GIDname, ".post.res" );
	strcpy( SURFname, argv[1] );
	strcat( SURFname, ".matlab.txt" );
	strcpy( TRIAXname, argv[1] );
	strcat( TRIAXname, ".triax.txt" );
	
	
	//deleting previuos files
	printf("Removing old files... ");
	lincr = 0L;
	do
	{
		lincr += 1L;
		strcpy( INCname, OUTname );
		strcpy( workstr, "." );
		if( lincr <100L ) strcat( workstr, "0" );
		if( lincr <10L ) strcat( workstr, "0" );
		ltoa( lincr, slincr, 10 );
		strcat( workstr, slincr );
		strcat( INCname, workstr );
		strcat( INCname, ".OUT.DAT" );
		if( remove( INCname ) != 0) break;
	}
	while( lincr<1000L );
	printf("OK\n\n");

	printf( "Reading data: \n" );
	//INP file : reads all calculation parameters and controls
	printf("Reading main input data : %s\n", INPname );	
	done = Read_INP( INPname );
	if( done <= 0 )
	{
		printf("Error in INP reading (%d) :(\n", done);
		printf("Exit.\n");
		return -1;
	}
	printf("Material set(s)         : %d\n", done );
	printf("Material identification : %d\n", MatIdentType );
	printf("Load increments         : %.0lf\n", INCmax/delta );
	printf("Reduction factor        : %.2lf\n", Fsafety );
	printf("Violation criterion     : %.2lf%%\n", Vcrit * (double) 100.0 );
	printf("Criterion limit         : %E\n", Fmin );
	printf("Substeps tolerance      : %.5lf%%\n", SStol * (double) 100.0 );
	if( SSmax < 10L )
	{
		printf("Warning! SSmax cannot be less then 10.\n");
		printf("Setting SSmax = 10.\n");
		SSmax = 10L;
	}
	printf("Max substeps            : %d\n", SSmax );
	printf("Error tolerance         : %.5lf%%\n", NRtol *(double) 100.0 );
	printf("Max iteration           : %d\n", NRmax );
	printf("Collapse error          : %.5lf%%\n", Collapse * (double) 100.0 );
	printf("Overrelaxation          : %.3lf%%\n", NRRelax );
	printf("Iteration procedure     : " );	
	if( VPuse == 1L ) printf("Viscoplastic\n"); else printf("Newton-Raphson\n");
	if(( VPuse == 1L ) and ( DEPuse == 1L ))
	{
		printf("Warning! In viscoplastic algorithm DEP cannot be used.\n");
		printf("Setting DEPuse = 0.\n");
		DEPuse = 0L;
	}
	printf("Tangent Dep use         : " );
	if( DEPuse == 1L ) printf("YES\n"); else printf("NO\n");

	printf("Automatic step size     : " );
	if( INCauto > 0L ) printf("YES (%d)\n", INCauto ); else printf("NO\n");
	printf("\n");

	//BMP file : soil positions and material parameters 
	printf("Reading bmp             : %s ", BMPname );
	done = Read_BMP( BMPname );
	if( done != 0 )
	{
		printf("Error in BMP reading (%d) :(\n", done);
		printf("Exit.\n");
		return -2;
	}
	printf("(%d)\n", done);
	printf("Width of BMP data       : %d\n", BMPwidth );
	printf("Height of BMP data      : %d\n", BMPheight );	
	printf("\n");
	
	//MSH file : reads mesh file (nodes, elements, constraints)
	printf("Reading mesh            : %s", MSHname);
	done = Read_MSH( MSHname );
	if( done != 0 )
	{
		printf("Error in MSH reading (%d) :(\n", done);
		switch( done )
		{
			case 1: printf("Cannot open MSH file!\n");
					break;
			case 2: printf("Too many nodes!\n");
					break;
			case 3: printf("Too many elements!\n");
					break;
			case 4: printf("Too many nodes constrained!\n");
					break;
		}
		printf("Exit.\n");
		return -3;
	}
	printf("(%d)\n", done);
	printf("Nodes                   : %d\n", NNODE );
	printf("Nodes to monitor        : %d\n", Nmonitor);
	printf("Unknowns created        : %d\n", NDIM );
	printf("Elements                : %d\n", NELEM );
	printf("Boundary conditions     : %d\n", NBND );	
	printf("Consolidation stress    : %.0lf\n", consolid_stress );

	printf("\nAll data read.\n\n" );

	/***********************
          CALCULATION
	************************/
	
	lincr = 0L;
	dincr = delta;
	for( i=0; i<NDIM; i++ )
	{
		u_act[i] = Zero;
		u_elas[i] = Zero;
		f_ext[i] = Zero;
		f_total[i] = f[i];
	}
	
	//saving GID postprocessor data
	Save4GID7( GIDname, lincr );
	
	//Reading initial stresses
	if( K0 == 1 ) 
	{		
		printf("\nTwo-stage calculation process.\n" );
		printf("First stage.\n" );
		for( i=0; i<NDIM; i++ ) df[i] = f_total[i];
		delta = INCmax;
		CalculateIt( );
		for( i=0; i<NDIM; i++ )
		{
			u_act[i] = Zero;
			f_total[i] = f[i];
			f[i] = Zero;
		}		
		printf("Second stage.\n" );
		SaveTriax( TRIAXname, 0L );
		Read_loads( LOADname );
		printf("\nExternal loads read           : OK\n" );
		Read_disp( DISPname );
		printf("Prescribed displacements read : OK\n" );
		delta = dincr;
		for( i=0; i<NDIM; i++ )
		{
			df[i] = f[i] / ( INClimit / delta );
			f[i] = f_total[i];
		}
		CalculateIt( );
	}
	else
	{
		printf("\nOne-stage calculation process.\n" );
		for( i=0; i<NDIM; i++ ) df[i] = f_total[i] / ( INClimit / delta );
		CalculateIt( );
	}
	
	//saving final strains & stresses in separate files
	//Save_eps( EPSname );
	Save_sig( SIGname );

	return 0;
}

