/*
	File			:	T6.CPP
	Identification	:	6 nodes, triangle element
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	14.07.2006
	Last modified	:	15.07.2006
	Content			:	CenterT6, NT6, GaussT6, mkT6, mkBT6, KmT6, esT6, rlT6, N6
*/

#ifndef _srokosz_T6_cpp
#define _srokosz_T6_cpp

/* BEGIN T6 */

#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include "FEMVars.h"
#include "FEMLib.h"
#include "MCPL.H"
#include "MMC.H"

static double B[4][12];

//-------------------------------------------------------------------------------------
//shape functions for T6
COOR4 NT6( int iGPoint )
{
	COOR4 N;
	double L1, L2, L3;
	
	L1 = GPT6[iGPoint].ksi;
	L2 = GPT6[iGPoint].eta;
	L3 = One - L1 - L2;

	N.one = ( Two * L1 - One ) * L1;
	N.two = Four * L3 * L1;
	N.three = ( Two * L3 - One ) * L3;
	N.four = Four * L2 * L3;
	N.five = ( Two * L2 - One ) * L2;
	N.six = Four * L1 * L2;
	return N;

}//NT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//center point of T3 element in global coordinates
//used for material constants identification
COOR2 CenterT6( long iElem )
{
	COOR2 G, P1, P2, P3, P4, P5, P6;
	COOR4 N;
	double L1, L2, L3;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	P5 = NODE[ELEM[iElem].node[4]];
	P6 = NODE[ELEM[iElem].node[5]];
	
	L1 = One / Three;
	L2 = One / Three;
	L3 = One - L1 - L2;

	N.one = ( Two * L1 - One ) * L1;
	N.two = Four * L3 * L1;
	N.three = ( Two * L3 - One ) * L3;
	N.four = Four * L2 * L3;
	N.five = ( Two * L2 - One ) * L2;
	N.six = Four * L1 * L2;

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x + N.five * P5.x + N.six * P6.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y + N.five * P5.y + N.six * P6.y;
	return G;

}//CenterT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//transformation : local to global coordinates
// used for Gauss integration points: stress&strains location
COOR2 GaussT6( long iElem, int iGPoint )
{
	COOR2 G, P1, P2, P3, P4, P5, P6;
	COOR4 N;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];
	P4 = NODE[ELEM[iElem].node[3]];
	P5 = NODE[ELEM[iElem].node[4]];
	P6 = NODE[ELEM[iElem].node[5]];
	
	N = NT6( iGPoint );

	G.x = N.one * P1.x + N.two * P2.x + N.three * P3.x + N.four * P4.x + N.five * P5.x + N.six * P6.x;
	G.y = N.one * P1.y + N.two * P2.y + N.three * P3.y + N.four * P4.y + N.five * P5.y + N.six * P6.y;
	return G;

}//GaussT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//making element
void mkT6( long iElem, long iNode1, long iNode2, long iNode3, long iNode4, long iNode5,
		    long iNode6 )
{
	int i;
	
	NELEM++;
	ELEM[iElem].typ = 6;
	ELEM[iElem].node[0] = iNode1;
	ELEM[iElem].node[1] = iNode2;
	ELEM[iElem].node[2] = iNode3;
	ELEM[iElem].node[3] = iNode4;
	ELEM[iElem].node[4] = iNode5;
	ELEM[iElem].node[5] = iNode6;
	ELEM[iElem].matpars = MATPAR_ZERO;
	for( i=0L; i<9L; i++ )
	{
		ELEM[iElem].Fmc[i] = NOne;
		ELEM[iElem].eps[i] = COOR3_ZERO;
		ELEM[iElem].sig[i] = COOR3_ZERO;
		ELEM[iElem].dsig[i] = COOR3_ZERO;
		ELEM[iElem].deps[i] = COOR3_ZERO;
		ELEM[iElem].depsvp[i] = COOR3_ZERO;
		ELEM[iElem].dsigvp[i] = COOR3_ZERO;
		ELEM[iElem].dQ[i] = COOR3_ZERO;
		ELEM[iElem].Gauss[i] = COOR2_ZERO;
		ELEM[iElem].u[i] = Zero;
		ELEM[iElem].plas[i] = 0L;
		ELEM[iElem].Edp[i]  = Zero;
	}

}//mkT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//T6 element : B matrix
double mkBT6( long iE, long iG )
{
	double J[2][2], InvJ[2][2], DetJ;
	double L1, L2;
	double dN[2][6], b[2][6];
	long row, col, i;
	COOR2 P1, P2, P3, P4, P5, P6;

	for( row=0; row<4; row++ )
	{
		for( col=0; col<12; col++ )
			B[row][col] = Zero;
	}

	P1 = NODE[ELEM[iE].node[0]];
	P2 = NODE[ELEM[iE].node[1]];
	P3 = NODE[ELEM[iE].node[2]];
	P4 = NODE[ELEM[iE].node[3]];
	P5 = NODE[ELEM[iE].node[4]];
	P6 = NODE[ELEM[iE].node[5]];

	/*
	y
	|
	|       P3
	|	  /    \
	|	P1------P5
	 ------------------x

	L2
	|
	0 P5
	|\
	|  \
	0---0---------------L1	
	P3  P1
	*/

	//coordinates of Gauss point
	L1 = GPT6[iG].ksi;
	L2 = GPT6[iG].eta;

	//derivatives dN/dlocal
	dN[0][0] = Four * L1 - One;
	dN[0][1] = Four * ( One - Two * L1 - L2 );
	dN[0][2] = Four * ( L1 + L2 ) - Three;
	dN[0][3] = NOne * Four * L2;
	dN[0][4] = Zero;
	dN[0][5] = Four * L2;
	dN[1][0] = Zero;
	dN[1][1] = NOne * Four * L1;
	dN[1][2] = Four * ( L1 + L2 ) - Three;
	dN[1][3] = Four * ( One - L1 - Two * L2 );
	dN[1][4] = Four * L2 - One;
	dN[1][5] = Four * L1;

	// Jacobi matrix = dN/dLocal * P(coor)
	J[0][0] = dN[0][0] * P1.x + dN[0][1] * P2.x + dN[0][2] * P3.x + dN[0][3] * P4.x + dN[0][4] * P5.x + dN[0][5] * P6.x;
	J[0][1] = dN[0][0] * P1.y + dN[0][1] * P2.y + dN[0][2] * P3.y + dN[0][3] * P4.y + dN[0][4] * P5.y + dN[0][5] * P6.y;
	J[1][0] = dN[1][0] * P1.x + dN[1][1] * P2.x + dN[1][2] * P3.x + dN[1][3] * P4.x + dN[1][4] * P5.x + dN[1][5] * P6.x;
	J[1][1] = dN[1][0] * P1.y + dN[1][1] * P2.y + dN[1][2] * P3.y + dN[1][3] * P4.y + dN[1][4] * P5.y + dN[1][5] * P6.y;
		
	//determinant of Jacobi matrix - Jacobian
	DetJ = J[0][0] * J[1][1] - J[0][1] * J[1][0];

	//inverted Jacobi matrix
	InvJ[0][0] = J[1][1] / DetJ;
	InvJ[0][1] = NOne * J[0][1] / DetJ;
	InvJ[1][0] = NOne * J[1][0] / DetJ;
	InvJ[1][1] = J[0][0] / DetJ;

	//B matrix
	for( row=0; row<2; row++ )
	{
		for( col=0; col<6; col++ )
		{
			b[row][col] = Zero;
			for( i=0; i<2; i++ )
				b[row][col] += InvJ[row][i] * dN[i][col];
		}
	}
	B[0][0] = b[0][0]; 
	B[0][2] = b[0][1]; 
	B[0][4] = b[0][2]; 
	B[0][6] = b[0][3];
	B[0][8] = b[0][4];
	B[0][10] = b[0][5];
	B[1][1] = b[1][0]; 
	B[1][3] = b[1][1]; 
	B[1][5] = b[1][2];
	B[1][7] = b[1][3];		
	B[1][9] = b[1][4];		
	B[1][11] = b[1][5];		
	B[2][0] = B[1][1]; 
	B[2][1] = B[0][0]; 
	B[2][2] = B[1][3]; 
	B[2][3] = B[0][2]; 
	B[2][4] = B[1][5]; 
	B[2][5] = B[0][4]; 
	B[2][6] = B[1][7]; 
	B[2][7] = B[0][6]; 
	B[2][8] = B[1][9]; 
	B[2][9] = B[0][8]; 
	B[2][10] = B[1][11]; 
	B[2][11] = B[0][10]; 

	return DetJ;

}//mkBT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//T6 element - stiffness
void KmT6( long iElem )
{
	double BTD[12][3], BTDB[12][12], km[12][12], DetJ, Weight;
	long row, col, i, j, lrow, lcol, ixy, jxy, iGauss;

	//zeroing km
	for( row=0; row<12; row++ )
	{
		for( col=0; col<12; col++ )
			km[row][col] = Zero;
	}
	
	//for each Gauss point
	for( iGauss=0; iGauss<3; iGauss++ )
	{		
		STRESS[0] = ELEM[iElem].sig[iGauss].x;
		STRESS[1] = ELEM[iElem].sig[iGauss].y;
		STRESS[2] = ELEM[iElem].sig[iGauss].xy;
		STRESS[3] = ELEM[iElem].sig[iGauss].z;

		if( DEPuse == 1L )
			mkDEP( ELEM[iElem].plas[iGauss], STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		else
			mkDEP( 0L, STRESS, ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars  );

		//coordinates of Gauss point
		Weight = GPT6[iGauss].w;

		//B matrix, DetJ
		DetJ = mkBT6( iElem, iGauss );

		//stiffnes matrix BTDB	
		//first : multiplication BT*D
		//transposed B in multiplication
		for( row=0; row<12; row++ )
		{
			for( col=0; col<3; col++ )
			{
				BTD[row][col] = Zero;
				for( i=0; i<3; i++ ) 
					BTD[row][col] += B[i][row] * DEP[i][col];
			}
		}

		//second : multiplication BTD*B
		for( row=0; row<12; row++ )
		{
			for( col=0; col<12; col++ )
			{
				BTDB[row][col] = Zero;
				for( i=0; i<3; i++ )
					BTDB[row][col] += BTD[row][i] * B[i][col];
			}
		}

		//third : integration DetJ*Wi*BTDB
		for( row=0; row<12; row++ )
		{
			for( col=0; col<12; col++ )
					km[row][col] += DetJ * Weight * BTDB[row][col];
		}		
	}

	//update global stiffness matrix
	i = 0;
	for( lrow=0; lrow<6; lrow++ )
	{
		for( ixy=0; ixy<2; ixy++ )
		{
			row = 2 * ELEM[iElem].node[lrow] + ixy;
			j = 0;
			for( lcol=0; lcol<6; lcol++ )
			{
				for( jxy=0; jxy<2; jxy++ )
				{
					col = 2 * ELEM[iElem].node[lcol] + jxy;				
					Km[row][col] += km[i][j];
					j+=1;
				}
			}
			i+=1;
		}
	}	
}//KmT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//stresses & strains T6
void esT6( long iElem, int opt )
{
	double uT6[12], Fc, DetJ, Edp_new, sig[4];
	long col, iGauss;
	COOR2 U;
	COOR3 deps;
	
	//retrieving displacement increments
	U = readNODE( u, ELEM[iElem].node[0] );
	uT6[0] = U.x;
	uT6[1] = U.y;
	U = readNODE( u, ELEM[iElem].node[1] );
	uT6[2] = U.x;
	uT6[3] = U.y;
	U = readNODE( u, ELEM[iElem].node[2] );
	uT6[4] = U.x;
	uT6[5] = U.y;
	U = readNODE( u, ELEM[iElem].node[3] );
	uT6[6] = U.x;
	uT6[7] = U.y;
	U = readNODE( u, ELEM[iElem].node[4] );
	uT6[8] = U.x;
	uT6[9] = U.y;
	U = readNODE( u, ELEM[iElem].node[5] );
	uT6[10] = U.x;
	uT6[11] = U.y;
   
	//for each Gauss point
	for( iGauss=0; iGauss<3; iGauss++ )
	{		
		//B matrix, DetJ (not used here)
		DetJ = mkBT6( iElem, iGauss );

		//strains increment, de=B*du
		deps.x = Zero;
		deps.y = Zero;
		deps.xy = Zero;
		deps.z = Zero;		
		for( col=0; col<12; col++ ) 
		{
			deps.x  += B[0][col] * uT6[col];		
		    deps.y  += B[1][col] * uT6[col];		
		    deps.xy += B[2][col] * uT6[col];
		}		

		if( opt == 0 )
		{
			//stress increment (Newton-Raphson)
			Edp_new = calcDSIG( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].dsig[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );
		
			//updating incremental stresses & strains
			ELEM[iElem].deps[iGauss].x  += deps.x;
			ELEM[iElem].deps[iGauss].y  += deps.y;
			ELEM[iElem].deps[iGauss].xy += deps.xy;
			ELEM[iElem].deps[iGauss].z  += deps.z;
			ELEM[iElem].dsig[iGauss].x  += dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  += dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy += dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  += dSTRESS[3];
			//yield value
			sig[0] = ELEM[iElem].sig[iGauss].x + ELEM[iElem].dsig[iGauss].x;
			sig[1] = ELEM[iElem].sig[iGauss].y + ELEM[iElem].dsig[iGauss].y;
			sig[2] = ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy;
			sig[3] = ELEM[iElem].sig[iGauss].z + ELEM[iElem].dsig[iGauss].z;
			Fc = Fmc_MMC( sig, Fi_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ), c_MMC( ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars ));
		}//N-R
		else
		{
			//viscoplastic algorithm

			//minimum DT, dsig=>dSTRESS, dQ=>DQ 
			Fc = calcVP( deps, ELEM[iElem].sig[iGauss], ELEM[iElem].depsvp[iGauss], ELEM[iElem].Edp[iGauss], ELEM[iElem].matpars );

			//saving stresses & strains increments
			ELEM[iElem].deps[iGauss].x  = deps.x;
			ELEM[iElem].deps[iGauss].y  = deps.y;
			ELEM[iElem].deps[iGauss].xy = deps.xy;
			ELEM[iElem].deps[iGauss].z  = deps.z;
			
			ELEM[iElem].dsig[iGauss].x  = dSTRESS[0];
			ELEM[iElem].dsig[iGauss].y  = dSTRESS[1];
			ELEM[iElem].dsig[iGauss].xy = dSTRESS[2];
			ELEM[iElem].dsig[iGauss].z  = dSTRESS[3];

			//saving Q-derivatives
			ELEM[iElem].dQ[iGauss].x  = DQ[0];
			ELEM[iElem].dQ[iGauss].y  = DQ[1];
			ELEM[iElem].dQ[iGauss].xy = DQ[2];
			ELEM[iElem].dQ[iGauss].z  = DQ[3];			
		}//vp
		
		//updating markers
		ELEM[iElem].Fmc[iGauss] = Fc;
		if( Fc > ( NOne * Fmin) ) 
		{
			ELEM[iElem].plas[iGauss] = 1L;
		}
		else
		{
			ELEM[iElem].plas[iGauss] = 0L;
		}

		//pore pressure, u=Du*e
		//matrix D, U
		mkDU( ELEM[iElem].matpars.Ke );
		ELEM[iElem].u[iGauss] = ELEM[iElem].matpars.Ke * ( ELEM[iElem].eps[iGauss].x + ELEM[iElem].eps[iGauss].y );
		
		//Gauss point
		ELEM[iElem].Gauss[iGauss] = GaussT6( iElem, iGauss );		
	}//Gauss point
}//esT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//viscoplastic strain & stress increment
void vpT6( long iElem )
{
	double ddepsvp[4], dsigvp[4];
	long row, col, iGauss;

	mkDE( ELEM[iElem].matpars.E, ELEM[iElem].matpars.ni, ELEM[iElem].matpars.Ke );
	for( iGauss=0L; iGauss<3L; iGauss++ )
	{
		if( ELEM[iElem].Fmc[iGauss] > Zero)
		{
			//viscoplastic strain increment
			ddepsvp[0] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].x;
			ddepsvp[1] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].y;
			ddepsvp[2] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].xy;
			ddepsvp[3] = DT * ELEM[iElem].Fmc[iGauss] * ELEM[iElem].dQ[iGauss].z;

			//plastic deviatoric strain
			ELEM[iElem].Edp[iGauss] += DT * ELEM[iElem].Fmc[iGauss];
				
			//updating viscoplastic strain increment
			ELEM[iElem].depsvp[iGauss].x  += ddepsvp[0];
			ELEM[iElem].depsvp[iGauss].y  += ddepsvp[1];
			ELEM[iElem].depsvp[iGauss].xy += ddepsvp[2];
			ELEM[iElem].depsvp[iGauss].z  += ddepsvp[3];

			//viscoplastic stress increment
			for( row=0L; row<4L; row++ )
			{
				dsigvp[row] = Zero;
				for( col=0L; col<4L; col++ )
				{
					dsigvp[row] += DE[row][col] * ddepsvp[col];
				}
			}
		}//F>0
		else
		{
			for( row=0L; row<4L; row++ ) dsigvp[row] = Zero;
		}
			
		//saving viscoplastic stress increment
		ELEM[iElem].dsigvp[iGauss].x  = dsigvp[0];
		ELEM[iElem].dsigvp[iGauss].y  = dsigvp[1];
		ELEM[iElem].dsigvp[iGauss].xy = dsigvp[2];
		ELEM[iElem].dsigvp[iGauss].z  = dsigvp[3];
	}//iGauss
}//vpT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//body loads, new version as residual loads
void rlT6( long iElem, int opt )
{
	int iGauss, row;
	double DetJ, BTS[12], fR[12];

	//zeroing
	for( row=0; row<12; row++ ) fR[row] = Zero;
	
	//integration
	for( iGauss=0; iGauss<3; iGauss++ )
	{
		//B matrix
		DetJ = mkBT6( iElem, iGauss );

		if( opt == 0 )
		{
			//multiplication BT*(sig+dsig)
			for( row=0; row<12; row++ )
			{
				BTS[row]  = B[0][row] * ( ELEM[iElem].sig[iGauss].x  + ELEM[iElem].dsig[iGauss].x );
				BTS[row] += B[1][row] * ( ELEM[iElem].sig[iGauss].y  + ELEM[iElem].dsig[iGauss].y );
				BTS[row] += B[2][row] * ( ELEM[iElem].sig[iGauss].xy + ELEM[iElem].dsig[iGauss].xy );
				BTS[row] += B[3][row] * ( ELEM[iElem].sig[iGauss].z  + ELEM[iElem].dsig[iGauss].z );
			}
		}
		else
		{
			//multiplication BT*dsigvp
			for( row=0; row<12; row++ )
			{
				BTS[row]  = B[0][row] * ELEM[iElem].dsigvp[iGauss].x;
				BTS[row] += B[1][row] * ELEM[iElem].dsigvp[iGauss].y;
				BTS[row] += B[2][row] * ELEM[iElem].dsigvp[iGauss].xy;
				BTS[row] += B[3][row] * ELEM[iElem].dsigvp[iGauss].z;
			}
		}
		
		//integration
		for( row=0; row<12; row++ )
		{
			fR[row] += DetJ * GPT6[iGauss].w * BTS[row];
		}
	}
	
	updatefR( ELEM[iElem].node[0], fR[0], fR[1] );
	updatefR( ELEM[iElem].node[1], fR[2], fR[3] );
	updatefR( ELEM[iElem].node[2], fR[4], fR[5] );
	updatefR( ELEM[iElem].node[3], fR[6], fR[7] );
	updatefR( ELEM[iElem].node[4], fR[8], fR[9] );
	updatefR( ELEM[iElem].node[5], fR[10], fR[11] );

}//rlT6
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double N6( long iElem, int iNode )
{
	int i;
	double vN, NT, DetJ;
	
	vN = Zero;
	for(i=0; i<3; i++)
	{
		switch(iNode)
		{
			case 1: NT = NT6(i).one;
					break;
			case 2: NT = NT6(i).two;
					break;
			case 3:	NT = NT6(i).three;
					break;
			case 4:	NT = NT6(i).four;
					break;
			case 5: NT = NT6(i).five;
					break;
			case 6: NT = NT6(i).six;
					break;
		}
		DetJ = mkBT6( iElem, i );
		vN += DetJ * GPT6[i].w * NT;
	}
	return vN;
}//T6
//-------------------------------------------------------------------------------------


/* END T6 */

#endif
