/*
	File			:	T3.CPP
	Identification	:	3 nodes, triangle element
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	14.07.2006
	Last modified	:	15.07.2006
	Content			:	CenterT3, GaussT3, mkT3, mkBT3, KmT3, esT3, rlT3, N3
*/

#ifndef _srokosz_T3_cpp
#define _srokosz_T3_cpp

/* BEGIN T3 */

#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <float.h>
#include <stdio.h>
#include "FEMVars.h"
#include "FEMLib.h"
#include "MCPL.H"
#include "MMC.H"

static double B[4][6];

//-------------------------------------------------------------------------------------
//center point of T3 element in global coordinates
//used for material constants identification
COOR2 CenterT3( long iElem )
{
	COOR2 G, P1, P2, P3;
	double OT;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];

	OT = One / Three;
	G.x = OT * P1.x + OT * P2.x + OT * P3.x;
	G.y = OT * P1.y + OT * P2.y + OT * P3.y;
	return G;

}//CenterT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//transformation : local to global coordinates
// used for Gauss integration points: stress&strains location
COOR2 GaussT3( long iElem )
{
	COOR2 G, P1, P2, P3;
	double OT;

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];

	OT = One / Three;
	G.x = OT * P1.x + OT * P2.x + OT * P3.x;
	G.y = OT * P1.y + OT * P2.y + OT * P3.y;
	return G;

}//GaussT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//making element
void mkT3( long iElem, long iNode1, long iNode2, long iNode3 )
{
	int i;
	
	NELEM++;
	ELEM[iElem].typ = 3;
	ELEM[iElem].node[0] = iNode1;
	ELEM[iElem].node[1] = iNode2;
	ELEM[iElem].node[2] = iNode3;
	ELEM[iElem].matpars = MATPAR_ZERO;
	for( i=0L; i<9L; i++ )
	{
		ELEM[iElem].Fmc[i] = NOne;
		ELEM[iElem].eps[i] = COOR3_ZERO;
		ELEM[iElem].sig[i] = COOR3_ZERO;
		ELEM[iElem].dsig[i] = COOR3_ZERO;
		ELEM[iElem].deps[i] = COOR3_ZERO;
		ELEM[iElem].depsvp[i] = COOR3_ZERO;
		ELEM[iElem].dsigvp[i] = COOR3_ZERO;
		ELEM[iElem].dQ[i] = COOR3_ZERO;
		ELEM[iElem].Gauss[i] = COOR2_ZERO;
		ELEM[iElem].u[i] = Zero;
		ELEM[iElem].plas[i] = 0L;
		ELEM[iElem].Edp[i]  = Zero;
	}

}//mkT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//making B matrix
double mkBT3( long iElem )
{
	double J[2][2], InvJ[2][2], DetJ;
	COOR2 P1, P2, P3;
	long row, col;

	for( row=0; row<4; row++ )
	{
		for( col=0; col<6; col++ )
			B[row][col] = Zero;
	}

	P1 = NODE[ELEM[iElem].node[0]];
	P2 = NODE[ELEM[iElem].node[1]];
	P3 = NODE[ELEM[iElem].node[2]];

	/*
	y
	|
	|       P2
	|	  /    \
	|	P1------P3
	 ------------------x

	L2
	|
	0 P3
	|\
	|  \
	0---0---------------L1	
	P2  P1
	*/

    // Jacobi matrix
	J[0][0] = P1.x - P2.x;
	J[0][1] = P1.y - P2.y;
	J[1][0] = P3.x - P2.x;
	J[1][1] = P3.y - P2.y;
	
	//determinant of Jacobi matrix - Jacobian
	DetJ = J[0][0] * J[1][1] - J[0][1] * J[1][0];

	//inverted Jacobi matrix
	InvJ[0][0] = J[1][1] / DetJ;
	InvJ[0][1] = NOne * J[0][1] / DetJ;
	InvJ[1][0] = NOne * J[1][0] / DetJ;
	InvJ[1][1] = J[0][0] / DetJ;
	
    //B matrix
	B[0][0] = InvJ[0][0]; 
	B[0][2] = NOne * ( InvJ[0][0] + InvJ[0][1] ); 
	B[0][4] = InvJ[0][1]; 
	B[1][1] = InvJ[1][0]; 
	B[1][3] = NOne * ( InvJ[1][0] + InvJ[1][1] ); 
	B[1][5] = InvJ[1][1]; 
	B[2][0] = B[1][1]; 
	B[2][1] = B[0][0]; 
	B[2][2] = B[1][3]; 
	B[2][3] = B[0][2]; 
	B[2][4] = B[1][5]; 
	B[2][5] = B[0][4];

	return DetJ;
	
}//mkBT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//T3 element - stiffness
void KmT3( long iElem )
{
	double BTD[6][3], BTDB[6][6], km[6][6], DetJ;
	long row, col, i, j, lrow, lcol, ixy, jxy;

	//zeroing km
	for( row=0; row<6; row++ )
	{
		for( col=0; col<6; col++ )
			km[row][col] = Zero;
	}

	STRESS[0] = ELEM[iElem].sig[0].x;
	STRESS[1] = ELEM[iElem].sig[0].y;
	STRESS[2] = ELEM[iElem].sig[0].xy;
	STRESS[3] = ELEM[iElem].sig[0].z;

	if( DEPuse == 1L )
		mkDEP( ELEM[iElem].plas[0], STRESS, ELEM[iElem].Edp[0], ELEM[iElem].matpars );
	else
		mkDEP( 0L, STRESS, ELEM[iElem].Edp[0], ELEM[iElem].matpars  );
	
	//matrix B, DetJ
	DetJ = mkBT3( iElem );

    //stiffnes matrix BTDB	
	//first : multiplication BT*D
	//transposed B in multiplication
	for( row=0; row<6; row++ )
	{
		for( col=0; col<3; col++ )
		{
			BTD[row][col] = Zero;
			for( i=0; i<3; i++ ) BTD[row][col] += B[i][row] * DEP[i][col];
		}
	}
	
	//second : multiplication BTD*B
	for( row=0; row<6; row++ )
	{
		for( col=0; col<6; col++ )
		{
			BTDB[row][col] = Zero;
			for( i=0; i<3; i++ ) BTDB[row][col] += BTD[row][i] * B[i][col];
		}
	}	

	//third : integration DetJ*Wi*BTDB, W=0.5
	for( row=0; row<6; row++ )
	{
		for( col=0; col<6; col++ )
			km[row][col] = DetJ * Half * BTDB[row][col];
	}

	//update global stiffness matrix
	i = 0;
	for( lrow=0; lrow<3; lrow++ )
	{
		for( ixy=0; ixy<2; ixy++ )
		{
			row = 2 * ELEM[iElem].node[lrow] + ixy;
			j = 0;
			for( lcol=0; lcol<3; lcol++ )
			{
				for( jxy=0; jxy<2; jxy++ )
				{
					col = 2 * ELEM[iElem].node[lcol] + jxy;				
					Km[row][col] += km[i][j];
					j+=1;
				}
			}
			i+=1;
		}
	}

}//KmT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//stresses & strains T3
void esT3( long iElem, int opt )
{
	double uT3[6], Fc, DetJ, Edp_new, sig[4];
	long col;
	COOR2 U;
	COOR3 deps;
	
	//retrieving displacement increments
	U = readNODE( u, ELEM[iElem].node[0] );
	uT3[0] = U.x;
	uT3[1] = U.y;
	U = readNODE( u, ELEM[iElem].node[1] );
	uT3[2] = U.x;
	uT3[3] = U.y;
	U = readNODE( u, ELEM[iElem].node[2] );
	uT3[4] = U.x;
	uT3[5] = U.y;
   
	//B matrix, DetJ (not used here)
	DetJ = mkBT3( iElem );

	//strains increment, de=B*du
	deps.x = Zero;
	deps.y = Zero;
	deps.xy = Zero;
	deps.z = Zero;		
	for( col=0L; col<6L; col++ ) 
	{
		deps.x  += B[0][col] * uT3[col];		
	    deps.y  += B[1][col] * uT3[col];		
	    deps.xy += B[2][col] * uT3[col];
	}		
		
	if( opt == 0 )
	{
		//stress increment (Newton-Raphson)
		Edp_new = calcDSIG( deps, ELEM[iElem].sig[0], ELEM[iElem].dsig[0], ELEM[iElem].Edp[0], ELEM[iElem].matpars );
	
		//updating incremental stresses & strains
		ELEM[iElem].deps[0].x  += deps.x;
		ELEM[iElem].deps[0].y  += deps.y;
		ELEM[iElem].deps[0].xy += deps.xy;
		ELEM[iElem].deps[0].z  += deps.z;
		ELEM[iElem].dsig[0].x  += dSTRESS[0];
		ELEM[iElem].dsig[0].y  += dSTRESS[1];
		ELEM[iElem].dsig[0].xy += dSTRESS[2];
		ELEM[iElem].dsig[0].z  += dSTRESS[3];
		//yield value
		sig[0] = ELEM[iElem].sig[0].x + ELEM[iElem].dsig[0].x;
		sig[1] = ELEM[iElem].sig[0].y + ELEM[iElem].dsig[0].y;
		sig[2] = ELEM[iElem].sig[0].xy + ELEM[iElem].dsig[0].xy;
		sig[3] = ELEM[iElem].sig[0].z + ELEM[iElem].dsig[0].z;
		Fc = Fmc_MMC( sig, Fi_MMC( ELEM[iElem].Edp[0], ELEM[iElem].matpars ), c_MMC( ELEM[iElem].Edp[0], ELEM[iElem].matpars ));
	}//N-R
	else
	{
		//viscoplastic algorithm

		//minimum DT, dsig=>dSTRESS, dQ=>DQ 
		Fc = calcVP( deps, ELEM[iElem].sig[0], ELEM[iElem].depsvp[0], ELEM[iElem].Edp[0], ELEM[iElem].matpars );

		//saving stresses & strains increments
		ELEM[iElem].deps[0].x  = deps.x;
		ELEM[iElem].deps[0].y  = deps.y;
		ELEM[iElem].deps[0].xy = deps.xy;
		ELEM[iElem].deps[0].z  = deps.z;
			
		ELEM[iElem].dsig[0].x  = dSTRESS[0];
		ELEM[iElem].dsig[0].y  = dSTRESS[1];
		ELEM[iElem].dsig[0].xy = dSTRESS[2];
		ELEM[iElem].dsig[0].z  = dSTRESS[3];

		//saving Q-derivatives
		ELEM[iElem].dQ[0].x  = DQ[0];
		ELEM[iElem].dQ[0].y  = DQ[1];
		ELEM[iElem].dQ[0].xy = DQ[2];
		ELEM[iElem].dQ[0].z  = DQ[3];			
	}//vp
		
	//updating markers
	ELEM[iElem].Fmc[0] = Fc;
	if( Fc > ( NOne * Fmin) ) 
	{
		ELEM[iElem].plas[0] = 1L;
	}
	else
	{
		ELEM[iElem].plas[0] = 0L;
	}

	//pore pressure, u=Du*e
	//matrix D, U
	mkDU( ELEM[iElem].matpars.Ke );
	ELEM[iElem].u[0] = ELEM[iElem].matpars.Ke * ( ELEM[iElem].eps[0].x + ELEM[iElem].eps[0].y );
		
	//Gauss point
	ELEM[iElem].Gauss[0] = GaussT3( iElem );		
}//esT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//viscoplastic strain & stress increment
void vpT3( long iElem )
{
	double ddepsvp[4], dsigvp[4];
	long row, col;

	mkDE( ELEM[iElem].matpars.E, ELEM[iElem].matpars.ni, ELEM[iElem].matpars.Ke );
	if( ELEM[iElem].Fmc[0] > Zero)
	{
		//viscoplastic strain increment
		ddepsvp[0] = DT * ELEM[iElem].Fmc[0] * ELEM[iElem].dQ[0].x;
		ddepsvp[1] = DT * ELEM[iElem].Fmc[0] * ELEM[iElem].dQ[0].y;
		ddepsvp[2] = DT * ELEM[iElem].Fmc[0] * ELEM[iElem].dQ[0].xy;
		ddepsvp[3] = DT * ELEM[iElem].Fmc[0] * ELEM[iElem].dQ[0].z;

		//plastic deviatoric strain
		ELEM[iElem].Edp[0] += DT * ELEM[iElem].Fmc[0];
				
		//updating viscoplastic strain increment
		ELEM[iElem].depsvp[0].x  += ddepsvp[0];
		ELEM[iElem].depsvp[0].y  += ddepsvp[1];
		ELEM[iElem].depsvp[0].xy += ddepsvp[2];
		ELEM[iElem].depsvp[0].z  += ddepsvp[3];

		//viscoplastic stress increment
		for( row=0L; row<4L; row++ )
		{
			dsigvp[row] = Zero;
			for( col=0L; col<4L; col++ )
			{
				dsigvp[row] += DE[row][col] * ddepsvp[col];
			}
		}
	}//F>0
	else
	{
		for( row=0L; row<4L; row++ ) dsigvp[row] = Zero;
	}
			
	//saving viscoplastic stress increment
	ELEM[iElem].dsigvp[0].x  = dsigvp[0];
	ELEM[iElem].dsigvp[0].y  = dsigvp[1];
	ELEM[iElem].dsigvp[0].xy = dsigvp[2];
	ELEM[iElem].dsigvp[0].z  = dsigvp[3];
}//vpT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//body loads, new version as residual loads
void rlT3( long iElem, int opt )
{
	int row;
	double DetJ, BTS[6], fR[6];

	//zeroing
	for( row=0; row<6; row++ ) fR[row] = Zero;
	
	//integration
	//B matrix
	DetJ = mkBT3( iElem );

	if ( opt == 0 )
	{
		//multiplication BT*(sig+dsig)
		for( row=0; row<6; row++ )
		{
			BTS[row]  = B[0][row] * ( ELEM[iElem].sig[0].x  + ELEM[iElem].dsig[0].x );
			BTS[row] += B[1][row] * ( ELEM[iElem].sig[0].y  + ELEM[iElem].dsig[0].y );
			BTS[row] += B[2][row] * ( ELEM[iElem].sig[0].xy + ELEM[iElem].dsig[0].xy );
			BTS[row] += B[3][row] * ( ELEM[iElem].sig[0].z  + ELEM[iElem].dsig[0].z );
		}
	}
	else
	{
		//multiplication BT*dsigvp
		for( row=0; row<6; row++ )
		{
			BTS[row]  = B[0][row] * ELEM[iElem].dsigvp[0].x;
			BTS[row] += B[1][row] * ELEM[iElem].dsigvp[0].y;
			BTS[row] += B[2][row] * ELEM[iElem].dsigvp[0].xy;
			BTS[row] += B[3][row] * ELEM[iElem].dsigvp[0].z;
		}
	}
		
	//integration
	for( row=0; row<6; row++ )
	{
		fR[row] += DetJ * Half * BTS[row];
	}
	
	updatefR( ELEM[iElem].node[0], fR[0], fR[1] );
	updatefR( ELEM[iElem].node[1], fR[2], fR[3] );
	updatefR( ELEM[iElem].node[2], fR[4], fR[5] );

}//rlT3
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double N3( long iElem )
{
	double vN, DetJ;
	
	DetJ = mkBT3( iElem );
	vN = DetJ * Half * One / Three;
	return vN;
}//T3
//-------------------------------------------------------------------------------------


/* END T3 */

#endif
