/*
	File			:	MMC.CPP
	Identification	:	Library of functions and procedures for FEM
						Mohr-Coulomb plasticity with hardening/softening
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	25.08.2006
	Last modified	:	25.08.2006
	Content			:	

	double Fi_MMC( double Edp, MATPAR mpars )
	double c_MMC( double Edp, MATPAR mpars )
	double Psi_MMC( double Edp, MATPAR mpars )
	double invarT_MMC( double sig[4] )
	double invarP_MMC( double sig[4] )
	double invarJ_MMC( double sig[4] )
	double invarJ3_MMC( double sig[4] )
	double g_MMC( double theta, double fipsi )
	double app_MMC( double sig[4], double fi, double psi, double c )
	double derivFFi_MMC( double sig[4], double Edp, MATPAR mpars )
	double derivFc_MMC( double sig[4], double Edp, MATPAR mpars )
	double derivFiE_MMC( double Edp, MATPAR mpars )
	double derivcE_MMC( double Edp, MATPAR mpars )
	double Fmc_MMC( double sig[4], double fi, double c )
	void derivInvar_MMC( double sig[4], double theta )
	void derivFQmc_MMC( double sig[4], double fi, double psi, double c )
	void derivFQdp_MMC( double theta, double fi, double psi )
	void DEP_MMC( double sig[4], double Edp, MATPAR mprs )
	double calcLAMBDA( double deps[4] )


*/

/******************************
     Procedures & functions
 ******************************/
#ifndef _srokosz_MMC_cpp
#define _srokosz_MMC_cpp

//system files
#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>

//FEM library files
#include "FEMLib.h"
#include "FEMVars.h"
#include "MCPL.H"
#include "mcdpl.h"

double dpds[4], dJds[4], dtds[4], dJ2ds[4], dJ3ds[4];

/*
sig	 :		0 - sig x
			1 - sig y
			2 - tau xy
			3 - sig z
*/

//-------------------------------------------------------------------------------------
//Internal friction angle inputs[deg] output[deg]
double Fi_MMC( double Edp, MATPAR mpars )
{
	double Fi;

	if( fabs( Edp ) < mpars.Edp_Fip1 ) 
	{
		Fi = mpars.Fii + ( mpars.Fip - mpars.Fii ) / mpars.Edp_Fip1 * fabs( Edp );
	}
	else
	{
		if( fabs( Edp ) < mpars.Edp_Fip2 )
		{
			Fi = mpars.Fip;
		}
		else
		{
			if( fabs( Edp ) < mpars.Edp_Fir )
			{
				Fi = mpars.Fip - ( mpars.Fip - mpars.Fir ) / ( mpars.Edp_Fir - mpars.Edp_Fip2 ) * ( fabs(Edp) - mpars.Edp_Fip2 );
			}
			else
			{
				Fi = mpars.Fir;
			}
		}
	}
	return Fi;
}//Fi_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Internal cohesion inputs[kPa] output[kPa]
double c_MMC( double Edp, MATPAR mpars )
{
	double c;

	if( fabs( Edp ) < mpars.Edp_cp1 ) 
	{
		c = mpars.ci + ( mpars.cp - mpars.ci ) / mpars.Edp_cp1 * fabs( Edp );
	}
	else
	{
		if( fabs( Edp ) < mpars.Edp_cp2 )
		{
			c = mpars.cp;
		}
		else
		{
			if( fabs( Edp ) < mpars.Edp_cr )
			{
				c = mpars.cp - ( mpars.cp - mpars.cr ) / ( mpars.Edp_cr - mpars.Edp_cp2 ) * ( fabs(Edp) - mpars.Edp_cp2 );
			}
			else
			{
				c = mpars.cr;
			}
		}
	}
	return c;
}//c_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Internal dilatance angle inputs[deg] output[deg]
double Psi_MMC( double Edp, MATPAR mpars )
{
	double Psi;

	if( fabs( Edp ) < mpars.Edp_Fip1 ) 
	{
		Psi = ( mpars.Fii + ( mpars.Fip - mpars.Fii ) / mpars.Edp_Fip1 * fabs( Edp )) * mpars.beta;
	}
	else
	{
		if( fabs( Edp ) < mpars.Edp_Fip2 )
		{
			Psi = mpars.Fip * mpars.beta;
		}
		else
		{
			if( fabs( Edp ) < mpars.Edp_Fir )
			{
				Psi =  mpars.beta * mpars.Fip - ( mpars.beta * mpars.Fip - mpars.Psir ) / ( mpars.Edp_Fir - mpars.Edp_Fip2 ) * ( fabs(Edp) - mpars.Edp_Fip2 );
			}
			else
			{
				Psi = mpars.Psir;
			}
		}
	}
	return Psi;

}//Psi_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double invarP_MMC( double sig[4] )
{
    double p;

	p = ( sig[0] + sig[1] + sig[3] ) / Three;
	return p;

}//invarP_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double invarJ2_MMC( double sig[4] )
{
	double J2;

	J2 = (( sig[0] - sig[1] ) * ( sig[0] - sig[1] ) + ( sig[1] - sig[3] ) * ( sig[1] - sig[3] ) + ( sig[3] - sig[0] ) * ( sig[3] - sig[0] )) / Six;
	J2 += sig[2] * sig[2];
	return J2;

}//invarJ3_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double invarJ_MMC( double sig[4] )
{
	double J2;

	J2 = invarJ2_MMC( sig );
	return sqrt( J2 );

}//invarJ_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double invarJ3_MMC( double sig[4] )
{
	double J3, p;

	p = invarP_MMC( sig );
	J3 = ( sig[0] - p ) * ( sig[1] - p ) * ( sig[3] - p ) - ( sig[3] - p ) * sig[2] * sig[2];
	return J3;

}//invarJ3_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Lode angle [rad]
double invarT_MMC( double sig[4] )
{
    double theta, J3, J, sine;

	J3 = invarJ3_MMC( sig );
	J = invarJ_MMC( sig );
	
	if( fabs(J) > (double) 1.0E-10 )
	{
		sine =  Three * sqrt( Three ) * J3 /( J * J * J * Two);
		if( sine > One ) sine = One;
		if( sine < NOne ) sine = NOne;
		theta = NOne * asin( sine ) / Three;
	}
	else
	{
		theta = Zero;
	}
	return theta;

}//invarT_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double g_MMC( double theta, double fipsi )
{
	double fipsirad, gt;

	fipsirad = deg2rad( fipsi );
	gt = sin( fipsirad ) / ( cos( theta ) - sin( theta ) * sin( fipsirad )/sqrt( Three ));
	return gt;

}//g_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double app_MMC( double sig[4], double fi, double psi, double c )
{
	double firad, gt, gpp, theta, p, a;

	firad = deg2rad( fi );
	theta = invarT_MMC( sig );
	gt = g_MMC( theta, fi );
	gpp = g_MMC( theta, psi );
	p = invarP_MMC( sig );
	a = ( c / tan( firad ) - p ) * gt / gpp + p;
	return a;

}//app_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//dF/dFi derivative
double derivFFi_MMC( double sig[4], double Edp, MATPAR mpars )
{
	double dFFi, theta, p, fi, c, sinf, cosf, cost, sint, sqrt3;

	theta = invarT_MMC( sig );
	p = invarP_MMC( sig );
	fi = Fi_MMC( Edp, mpars );
	sinf = sin( deg2rad( fi ));
	cosf = cos( deg2rad( fi ));
	sint = sin( theta );
	cost = cos( theta );
	c = c_MMC( Edp, mpars );
	sqrt3 = sqrt( Three );
	dFFi = cost * ( p * cosf + c * sinf ) - c * sint / sqrt3;
	dFFi /= cost - sint * sinf / sqrt3; 
	dFFi /= cost - sint * sinf / sqrt3; 
	
	return dFFi;

}//derivFi_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//dF/dc derivative
double derivFc_MMC( double sig[4], double Edp, MATPAR mpars )
{
	double dFc, theta, fi, tanf, gt;

	theta = invarT_MMC( sig );
	fi = Fi_MMC( Edp, mpars );
	tanf = tan( deg2rad( fi ));
	gt = g_MMC( theta, fi );
	dFc = NOne * gt / tanf;
	
	return dFc;

}//derivFc_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//dFi/dE derivative (calcs in [rad])
double derivFiE_MMC( double Edp, MATPAR mpars )
{
	double dFi;

	if( fabs( Edp ) < mpars.Edp_Fip1 ) 
	{
		dFi = ( deg2rad(mpars.Fip) - deg2rad(mpars.Fii) ) / mpars.Edp_Fip1;
	}
	else
	{
		if( fabs( Edp ) < mpars.Edp_Fip2 )
		{
			dFi = Zero;
		}
		else
		{
			if( fabs( Edp ) < mpars.Edp_Fir )
			{
				dFi = ( deg2rad(mpars.Fir) - deg2rad(mpars.Fip) ) / ( mpars.Edp_Fir - mpars.Edp_Fip2 );
			}
			else
			{
				dFi = Zero;
			}
		}
	}

	return dFi;

}//derivFiE_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//dc/dE derivative
double derivcE_MMC( double Edp, MATPAR mpars )
{
	double dc;

	if( fabs( Edp ) < mpars.Edp_cp1 ) 
	{
		dc = ( mpars.cp - mpars.ci ) / mpars.Edp_cp1;
	}
	else
	{
		if( fabs( Edp ) < mpars.Edp_cp2 )
		{
			dc = Zero;
		}
		else
		{
			if( fabs( Edp ) < mpars.Edp_cr )
			{
				dc = ( mpars.cr - mpars.cp ) / ( mpars.Edp_cr - mpars.Edp_cp2 );
			}
			else
			{
				dc = Zero;
			}
		}
	}

	return dc;

}//derivcE_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double Fmc_MMC( double sig[4], double fi, double c )
{
	double firad, theta, gt, Fcrit, J, p;

	firad = deg2rad( fi );
	theta = invarT_MMC( sig );
	J = invarJ_MMC( sig );
	p = invarP_MMC( sig );
	gt = g_MMC( theta, fi );
	Fcrit = J - ( c / tan( firad ) - p ) * gt;
	return Fcrit;

}//Fmc_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//global p-J-teta/sig (dpds[4], dJds[4], dtds[4]) filled
void derivInvar_MMC( double sig[4] )
{
	double sx, sy, sz, p, J, sqrtJJ3, J3, sqrt3, theta, dtdJ, dtdJ3;
	double M3[4][4], M2[4][4], J2;
	long row, col;

	//invariants derivatives
	
	//invariants
	J2 = invarJ2_MMC( sig );
	J = sqrt( J2 );
	p = invarP_MMC( sig );
	J3 = invarJ3_MMC( sig );
	theta = invarT_MMC( sig );
	sqrt3 = sqrt( Three );
	
	//dp/ds
	dpds[0] = One / Three;
	dpds[1] = One / Three;
	dpds[2] = Zero;
	dpds[3] = One / Three;

	//dJ/ds
	dJds[0] = ( sig[0] - p ) / ( Two * J );
	dJds[1] = ( sig[1] - p ) / ( Two * J );
	dJds[2] = sig[2] / J;
	dJds[3] = ( sig[3] - p ) / ( Two * J );

	//dJ2/ds
	M2[0][0] = Two;
	M2[0][1] = NOne;
	M2[0][2] = Zero;
	M2[0][3] = NOne;

	M2[1][0] = NOne;
	M2[1][1] = Two;
	M2[1][2] = Zero;
	M2[1][3] = NOne;

	M2[2][0] = Zero;
	M2[2][1] = Zero;
	M2[2][2] = Six;
	M2[2][3] = Zero;
 
	M2[3][0] = NOne;
	M2[3][1] = NOne;
	M2[3][2] = Zero;
	M2[3][3] = Two;

	for( row=0L; row<4L; row++ )
	{
		dJ2ds[row] = Zero;
		for( col=0L; col<4L; col++ )
			dJ2ds[row] += M2[row][col] * sig[col] / Three;
	}
	
	//dteta/ds needs dJ/ds and dJ3/ds
	//M3 (derivative dJ3/ds)	
	sx = ( Two * sig[0] - sig[1] - sig[3] ) / Three;
	sy = ( Two * sig[1] - sig[3] - sig[0] ) / Three;
	sz = ( Two * sig[3] - sig[0] - sig[1] ) / Three;

	M3[0][0] = sx;
	M3[0][1] = sz;
	M3[0][2] = sig[2];
	M3[0][3] = sy;

	M3[1][0] = sz;
	M3[1][1] = sy;
	M3[1][2] = sig[2];
	M3[1][3] = sx;

	M3[2][0] = sig[2];
	M3[2][1] = sig[2];
	M3[2][2] = NOne * Three * sz;
	M3[2][3] = NOne * Two * sig[2];
 
	M3[3][0] = sy;
	M3[3][1] = sx;
	M3[3][2] = NOne * Two * sig[2];
	M3[3][3] = sz;

	for( row=0L; row<4L; row++ )
	{
		dJ3ds[row] = Zero;
		for( col=0L; col<4L; col++ )
			dJ3ds[row] += M3[row][col] * sig[col] / Three;
	}
		
	//dteta/ds
	sqrtJJ3 = sqrt( Four * J2 * J2 * J2 - (double) 27.0 * J3 * J3 );
	dtdJ3 = NOne * sqrt3 / sqrtJJ3;
	dtdJ = Three * sqrt3 * J3 / ( J * sqrtJJ3 );	
	for( row=0L; row<4L; row++ )
		dtds[row] = dtdJ3 * dJ3ds[row] + dtdJ * dJds[row];

}//derivInvar_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//global DF[4]+DQ[4] filled - MohrCoulomb
void derivFQmc_MMC( double sig[4], double fi, double psi, double c )
{
	double p, firad, psirad, denom, sinf, sint, cost, cosf, sinp, app, theta;
	double dFdp, dFdJ, dFdt, dQdp, dQdJ, dQdt, sqrt3, cos3t, tan3t, tant;
//	double dQdJ2, dQdJ3, J;
	long row;

	p = invarP_MMC( sig );
	theta = invarT_MMC( sig );
//	J = invarJ_MMC( sig );
	firad = deg2rad( fi );
	psirad = deg2rad( psi );
	sinf = sin( firad );
	sinp = sin( psirad );
	cosf = cos( firad );
	sint = sin( theta );
	cost = cos( theta );
	tant = tan( theta );
	cos3t = cos( Three * theta );
	tan3t = tan( Three * theta );
	sqrt3 = sqrt( Three );

	//F derivatives
	dFdp = g_MMC( theta, fi );
	dFdJ = One;
	denom = cost - sint * sinf / sqrt3;
	dFdt = ( p * sinf - c * cosf ) * ( sint + cost * sinf / sqrt3 ) / ( denom * denom );

	//global DF[4]
	for( row=0L; row<4L; row++ )
		DF[row] = dFdp * dpds[row] + dFdJ * dJds[row] + dFdt * dtds[row];

	//Q derivatives
	if( psi == Zero ) psi = nonZERO;//psi close to Zero	
	app = app_MMC( sig, fi, psi, c );		
	//Potts
	dQdp = g_MMC( theta, psi );
	dQdJ = One;
	denom = cost - sint * sinp / sqrt3;
	dQdt = ( p - app ) * sinp * ( sint + cost * sinp / sqrt3 ) / ( denom * denom );
	//global DQ[4]
	for( row=0L; row<4L; row++ )
		DQ[row] = dQdp * dpds[row] + dQdJ * dJds[row] + dQdt * dtds[row];
	/*
	{
		//Griffiths
		dQdp = sinp;
		dQdJ2 = cost / ( Two * J ) * ( One + tant * tan3t + sinp / sqrt3 * ( tan3t - tant ));
		dQdJ3 = ( sqrt3 * sint + sinp * cost ) / ( Two * J * J * cos3t );
		//global DQ[4]
		for( row=0L; row<4L; row++ )
			DQ[row] = dQdp * dpds[row] + dQdJ2 * dJ2ds[row] + dQdJ3 * dJ3ds[row];
	}
	*/	
	
}//derivFQmc_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//global DF[4]+DQ[4] filled - DruckerPrager
void derivFQdp_MMC( double theta, double fi, double psi )
{
	double firad, psirad, sinf, sint, sinp;
	double dFdp, dFdJ, dFdt, dQdp, dQdJ, dQdt, MJP, MJPpp, sqrt3;
	long row;

	firad = deg2rad( fi );
	psirad = deg2rad( psi );
	sinf = sin( firad );
	sinp = sin( psirad );
	sint = sin( theta );
	sqrt3 = sqrt( Three );

	if( sint < (double) -0.49 ) 
	{
		MJP = Two * sqrt3 * sinf / ( Three + sinf );
		MJPpp = Two * sqrt3 * sinp / ( Three + sinp );
	}
	if( sint > (double) 0.49 ) 
	{
		MJP = Two * sqrt3 * sinf / ( Three - sinf );
		MJPpp = Two * sqrt3 * sinp / ( Three - sinp );
	}
	
	//F derivatives
	dFdp = MJP;
	dFdJ = One;
	dFdt = Zero;

	//Q derivatives
	dQdp = MJPpp;
	dQdJ = One;
	dQdt = Zero;
	
	//global DF[4] and DQ[4]
	for( row=0L; row<4L; row++ )
	{
		DF[row] = dFdp * dpds[row] + dFdJ * dJds[row] + dFdt * dtds[row];
		DQ[row] = dQdp * dpds[row] + dQdJ * dJds[row] + dQdt * dtds[row];
	}

}//derivFQdp_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//global DEP[4][4] + DF[4] + DQ[4] + DEDQ[4] + DFDE[4] + DFDEDQ filled
void DEP_MMC( double sig[4], double Edp, MATPAR mprs )
{
	double theta, sint;
	long row, col;
	double fi, psi, c;

	fi = Fi_MMC( Edp, mprs );
	c = c_MMC( Edp, mprs );
	psi = Psi_MMC( Edp, mprs );
	theta = invarT_MMC( sig );
	sint = sin( theta );
	
	//invariants derivatives (global: dpds[4], dJds[4], dtds[4])
	derivInvar_MMC( sig );

	//F+Q derivatives
	if( fabs(sint) > (double) 0.49 )
	{
		derivFQdp_MMC( theta, fi, psi ); 
	}
	else
	{
		derivFQmc_MMC( sig, fi, psi, c );
	}
	
	for( row=0L; row<4L; row++ )
	{
		DFDE[row] = Zero;
		for( col=0L; col<4L; col++ )
		{
			DFDE[row] += DF[col] * DE[col][row];
		}
	}

	for( row=0L; row<4L; row++ )
	{
		DEDQ[row] = Zero;
		for( col=0L; col<4L; col++ )
		{
			DEDQ[row] += DE[row][col] * DQ[col];
		}
	}

	DFDEDQ = Zero;
	for( row=0L; row<4L; row++ ) DFDEDQ += DFDE[row] * DQ[row];

	for( row=0L; row<4L; row++ )
	{
		for( col=0L; col<4L; col++ )
		{
			DP[row][col] = DEDQ[row] * DFDE[col] / DFDEDQ;
			DEP[row][col] = DE[row][col] - DP[row][col];
		}
	}

	/*	
	//Full Griffiths: DF, DQ, DFDE, DEDQ, DFDEDQ
	GriffithsDEP( sig, fi, psi );
	*/

	
	//global A
	A = NOne * ( derivFFi_MMC( sig, Edp, mprs ) * derivFiE_MMC( Edp, mprs ) + derivFc_MMC( sig, Edp, mprs ) * derivcE_MMC( Edp, mprs ));
	
}//DEP_MMC
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double calcLAMBDA( double deps[4] )
{
	int row;
	double lambda;

	lambda = Zero;
	for(row=0; row<4; row++ )
	{
		lambda += DFDE[row] * deps[row];
	}
	lambda /= ( DFDEDQ + A );
	
	return lambda;

}//calcLAMBDA
//-------------------------------------------------------------------------------------


#endif

/* END */