/*
	File			:	MCPL.CPP
	Identification	:	Library of functions and procedures for FEM
						Mohr-Coulomb plasticity
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	12.07.2006
	Last modified	:	12.07.2006
	Content			:	

	void mkDEP( long fac, double sig[4], double Edp, MATPAR mprs )
	double findAlfa( double sig0[4], double dsig[4], double fi, double c )
	double correctSIG( double sig1[4], double sig2[4], double Edp, MATPAR mprs )



*/

#ifndef _srokosz_MCPL_cpp
#define _srokosz_MCPL_cpp

#include <iso646.h>
#include "FEMLib.h"
#include "MMC.H"
#define _USE_MATH_DEFINES
#include <math.h>
#include <stdio.h>
#include <float.h>

/* BEGIN */

double DP[4][4], DEP[4][4], NRctrl;
double DFDEDQ, DF[4], DQ[4], DEDQ[4], DFDE[4], A;
//control variables read from INP file
double Fmin;	//F equal "0.0"
double SStol;	//max error of substepping
long SSmax;		//max substepping iterations
double NRtol;	//Newton-Raphson tolerance error
long NRmax;		//max Newton-Raphson iterations
double INCmax;	//max load increments
long INCauto;	//flag of auto change increment size
long DEPuse;	//flag : use DEP or DE
long VPuse;		//flag : use Visco-plastic (1) or Newton-Raphson (0)
double Collapse;//max error indicating collaps
long InternCtrl;//internal control variable

/******************************
     Procedures & functions
 ******************************/


//-------------------------------------------------------------------------------------
//Elasto-Plastic matrix
//-------------------------------------------------------------------------------------
void mkDEP( long fac, double sig[4], double Edp, MATPAR mprs )
{
	long row, col;

	mkDE( mprs.E, mprs.ni, mprs.Ke );

	if( fac == 1L )
	{
		DEP_MMC( sig, Edp, mprs );
	}
	else
	{		
		for( row=0L; row<4L; row++ )
		{
			for( col=0L; col<4L; col++ )
			{
				DEP[row][col] = DE[row][col];
			}
		}
	}
}//mkDEP
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double findAlfa( double sig0[4], double dsig[4], double fi, double c )
{
	int row, iter, done;
	double F, Fprev, sig[4], sigprev[4], a, anext, aprev;

	aprev = Zero;
	a = One;
	iter = 0;
	done = 1;
	do
	{
		iter += 1;
		for( row=0; row<4; row++ ) 
		{
			sigprev[row] = sig0[row] + aprev * dsig[row];
			sig[row] = sig0[row] + a * dsig[row];
		}
		Fprev = Fmc_MMC( sigprev, fi, c );
		F = Fmc_MMC( sig, fi, c );
		if( fabs(F) < Fmin ) done = 0;
		if( iter > 30 ) done = 0;
		anext = a - F/( F - Fprev ) * ( a - aprev );
		aprev = a;
		a = anext;
	}
	while( done != 0 );
	if( F > Fmin ) printf("Alfa not found!\n");
	return a;

}//findAlfa
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//correction of stress state (F>0), return Edp
//return on plastic surface, results stored in STRESS vector
double correctSIG( double sig1[4], double sig2[4], double Edp, MATPAR mprs )
{
	long row, done, iter;
	double sigC[4], sigA[4], sigB[4];
	double lambda, Fini, FB, FC, Edp_new;

    //checking initial plastic criterion
	Fini = Fmc_MMC( sig2, Fi_MMC( Edp, mprs ), c_MMC( Edp, mprs ));
	
	//Fini<Fmin, nothing to do...
	if( Fini < Fmin )
	{
		for( row=0L; row<4L; row++ ) STRESS[row] = sig2[row];
		return Edp;
	}

	//rearranging vectors
	for( row=0L; row<4L; row++ )
	{
		sigA[row] = sig1[row];
		sigB[row] = sig2[row];
	}
	
	//correcting stress state: iterations	
	done = 1L;
	iter = 0L;
	Edp_new = Edp;
	FB = Fini;
	do
	{		
		iter += 1L;

		//all derivatives
		mkDEP( 1L, sigB, Edp_new, mprs );

		//lambda
		lambda = FB / ( DFDEDQ + A );

		//corrected stress, plastic deviatoric strain
		for( row=0L; row<4L; row++ ) sigC[row] = sigB[row] - lambda * DEDQ[row];
		Edp_new += lambda;

		//criterion
		FC = Fmc_MMC( sigC, Fi_MMC( Edp_new, mprs ), c_MMC( Edp_new, mprs ));

		//convergence
		if( fabs(FC) < Fmin ) done = 0L;
		if( iter > SSmax ) done = 0L;
		if( _isnan(FC) != 0 ) done = 0L;	
		
		//preparing for next iterations
		for( row=0L; row<4L; row++ ) sigB[row] = sigC[row];
		FB = FC;
	}
	while ( done > 0L );
	if( iter > InternCtrl ) InternCtrl = iter;
	if( _isnan(FC) != 0 ) return Free;	
	if( fabs(FC) > Fmin ) return Free;

	STRESS[0] = sigC[0];
	STRESS[1] = sigC[1];
	STRESS[2] = sigC[2];
	STRESS[3] = sigC[3];

	return Edp_new;

}//correctSIG
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//variable number of substeps (depending on Iter)
//NOT USED
long scaleItermax( void )
{
	
	long iters;
	double x, y, a3, a2, a1, a0;
	a3 = (double) 2.6182380249;
	a2 = (double) -3.9142324134;
	a1 = (double) -0.50305686663;
	a0 = Two;
	x = (double) Iter / (double) NRmax;	
	y = a3 * x * x * x + a2 * x * x + a1 * x + a0;	
	iters = (long) ( SSmax * y );
	if( iters < 2L ) iters = 2L;
	return iters;

}//scaleItermax
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//sliding on plastic surface, results stored in STRESS vector
//Runge-Kutta 4-th order method
double substepsRK( double sige[4], double deps[4], double Edp, MATPAR mprs )
{
	long row, col, iter, itermax, siter, ok_done;
	double sig1[4], sig2[4], sig3[4], sig4[4], sig0[4];
	double dsig1[4], dsig2[4], dsig3[4], dsig4[4], Edp1, Edp2, Edp3, Edp4;
	double dEdp1, dEdp2, dEdp3, dEdp4, dEdp, Err;
	double dsig[4], depss[4], Edp_cor, prevnsig, nextnsig;
 
	//init	
	itermax = SSmax;
	//step size iterations
	ok_done = 1L;
	siter = 0L;
	do
	{
		for( row=0; row<4; row++ ) 
		{
			sig1[row] = sige[row];		
			depss[row] = deps[row] / (double) itermax;
		}
		Edp1 = Edp;
		//substeps iterations
		iter = 0L;
		do
		{		
			for( row=0L; row<4L; row++ ) sig0[row] = sig1[row];		

			//x0, y0, k1
			mkDEP( 1L, sig1, Edp1, mprs );
			dEdp1 = calcLAMBDA( depss );
			Edp2 = Edp1 + Half * dEdp1;
			for( row=0L; row<4L; row++ )
			{
				dsig1[row] = Zero;
				for( col=0L; col<4L; col++ ) dsig1[row] += DEP[row][col] * depss[col];
				sig2[row] = sig1[row] + Half * dsig1[row];
			}

			//x0+h/2, y0+k1/2, k2
			mkDEP( 1L, sig2, Edp2, mprs );
			dEdp2 = calcLAMBDA( depss );
			Edp3 = Edp1 + Half * dEdp2;
			for( row=0L; row<4L; row++ )
			{
				dsig2[row] = Zero;
				for( col=0L; col<4L; col++ ) dsig2[row] += DEP[row][col] * depss[col];
				sig3[row] = sig1[row] + Half * dsig2[row];
			}

			//x0+h/2, y0+k2/2, k3
			mkDEP( 1L, sig3, Edp3, mprs );
			dEdp3 = calcLAMBDA( depss );
			Edp4 = Edp1 + dEdp3;
			for( row=0L; row<4L; row++ )
			{
				dsig3[row] = Zero;
				for( col=0L; col<4L; col++ ) dsig3[row] += DEP[row][col] * depss[col];
				sig4[row] = sig1[row] + dsig3[row];
			}

			// x0+h, y0+k3, k4
			mkDEP( 1L, sig4, Edp4, mprs );
			dEdp4 = calcLAMBDA( depss );
			for( row=0L; row<4L; row++ )
			{
				dsig4[row] = Zero;
				for( col=0L; col<4L; col++ ) dsig4[row] += DEP[row][col] * depss[col];
			}

			//final values
			for( row=0L; row<4L; row++ )
				dsig[row] = ( dsig1[row] + Two * dsig2[row] + Two * dsig3[row] + dsig4[row] ) / Six;		
			dEdp = ( dEdp1 + Two * dEdp2 + Two * dEdp3 + dEdp4 ) / Six;
				
			for( row=0L; row<4L; row++ ) sig1[row] += dsig[row];
			Edp1 += dEdp;
				
			//stress correction		
			InternCtrl = 0L;
			Edp_cor = correctSIG( sig0, sig1, Edp1, mprs );
			if( Edp_cor != Free )
			{
				for( row=0L; row<4L; row++ ) sig1[row] = STRESS[row];
				Edp1 = Edp_cor;
			}
			iter += 1L;
		}
		while( iter < itermax );
		if( siter > 0L )
		{
			//step size error
			nextnsig = normV( sig1, 4L );
			Err = fabs( prevnsig - nextnsig ) / nextnsig;
			prevnsig = nextnsig;
			if( Err < SStol ) ok_done = 0L;
		}
		else
		{
			prevnsig = normV( sig1, 4L );
		}
		siter += 1L;
		itermax *= 2L;
		if( siter > 10L ) ok_done = 0L;
	}
	while( ok_done > 0L );	
	
	if( InternCtrl > RetCtrl ) RetCtrl = InternCtrl;
	if( siter > IterCtrl ) IterCtrl = siter;
	//results
	for( row=0L; row<4L; row++ ) STRESS[row] = sig1[row];
	return Edp1;

}//substepsRK
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//calculates dsig (substepping+correction), results in dSTRESS
double calcDSIG( COOR3 deps, COOR3 sig0, COOR3 dsig0, double Edp, MATPAR mprs )
{
	long row;
	double Unload, Fcrit, stress[4], dsig[4], depsep[4], Fprev, fac, Edp_new;

	Edp_new = Edp;
	mkDE( mprs.E, mprs.ni, mprs.Ke );
	//stresses
	//checking yield criterion with elastic behaviour
	dsig[0] = DE[0][0] * deps.x + DE[0][1] * deps.y + DE[0][2] * deps.xy;
	dsig[1] = DE[1][0] * deps.x + DE[1][1] * deps.y + DE[1][2] * deps.xy;
	dsig[2] = DE[2][0] * deps.x + DE[2][1] * deps.y + DE[2][2] * deps.xy;
	dsig[3] = DE[3][0] * deps.x + DE[3][1] * deps.y + DE[3][2] * deps.xy;
	
	//actual stress state (including all previous dsig increments)
	stress[0] = sig0.x  + dsig0.x;
	stress[1] = sig0.y  + dsig0.y;
	stress[2] = sig0.xy + dsig0.xy;
	stress[3] = sig0.z  + dsig0.z;

	//new stress state
	for( row=0L; row<4L; row++ ) STRESS[row] = stress[row] + dsig[row];
		
	Fcrit = Fmc_MMC( STRESS, Fi_MMC( Edp, mprs ), c_MMC( Edp, mprs ));
	
	mkDEP( 1L, STRESS, Edp, mprs );

	Unload = Zero;
	for( row=0L; row<4L; row++ ) Unload += DF[row] * STRESS[row];
	
	//If it is elasticity, nothing to do... 
	
	//Plasticity and ideally on yield surface, nothing to do...
		
	//Oh no... outside yield surface!
	if( Fcrit > Fmin )
	{
		Fprev = Fmc_MMC( stress, Fi_MMC( Edp, mprs ), c_MMC( Edp, mprs ));
		//point became plastic
		if(( Fprev < Zero ) or ( Unload < Zero ))
		{
			//finding factor fac
			fac = findAlfa( stress, dsig, Fi_MMC( Edp, mprs ), c_MMC( Edp, mprs ));
			
			//elastic part of behaviour
			for( row=0L; row<4L; row++ ) stress[row] += fac * dsig[row];
			
			//elasto-plastic part of behaviour
			depsep[0] = ( One - fac ) * deps.x;
			depsep[1] = ( One - fac ) * deps.y;
			depsep[2] = ( One - fac ) * deps.xy;
			depsep[3] = ( One - fac ) * deps.z;

			//substeps algorithm, results in STRESS
			Edp_new = substepsRK( stress, depsep, Edp, mprs );				
		}//Fprev<0
		else
		{
			//point is already plastic
			//DEP instead of DE should be used...
			//substepping	
			depsep[0] = deps.x;
			depsep[1] = deps.y;
			depsep[2] = deps.xy;
			depsep[3] = deps.z;
			
			//output in STRESS
			Edp_new = substepsRK( stress, depsep, Edp, mprs );				
		}//Fprev=0
	}//Fcrit>Fmin			

	//common operations
	dSTRESS[0] = STRESS[0] - sig0.x  - dsig0.x;
	dSTRESS[1] = STRESS[1] - sig0.y  - dsig0.y;
	dSTRESS[2] = STRESS[2] - sig0.xy - dsig0.xy;
	dSTRESS[3] = STRESS[3] - sig0.z  - dsig0.z;
	return Edp_new;

}//calcDSIG
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//calculates dsig (elastic, dSTRESS), DT & DQ
double calcVP( COOR3 deps, COOR3 sig0, COOR3 depsvp0, double Edp, MATPAR mprs )
{
	double Fcrit, sig[4], ctrl, dsig[4];
	double dt1, dt2, dt;
	long row;

	//elastic stress increment
	mkDE( mprs.E, mprs.ni, mprs.Ke );

	dsig[0]  = DE[0][0] * ( deps.x  - depsvp0.x );
	dsig[0] += DE[0][1] * ( deps.y  - depsvp0.y );
	dsig[0] += DE[0][2] * ( deps.xy - depsvp0.xy );
	dsig[0] += DE[0][3] * ( deps.z  - depsvp0.z );

	dsig[1]  = DE[1][0] * ( deps.x  - depsvp0.x );
	dsig[1] += DE[1][1] * ( deps.y  - depsvp0.y );
	dsig[1] += DE[1][2] * ( deps.xy - depsvp0.xy );
	dsig[1] += DE[1][3] * ( deps.z  - depsvp0.z );

	dsig[2]  = DE[2][0] * ( deps.x  - depsvp0.x );
	dsig[2] += DE[2][1] * ( deps.y  - depsvp0.y );
	dsig[2] += DE[2][2] * ( deps.xy - depsvp0.xy );
	dsig[2] += DE[2][3] * ( deps.z  - depsvp0.z );

	dsig[3]  = DE[3][0] * ( deps.x  - depsvp0.x );
	dsig[3] += DE[3][1] * ( deps.y  - depsvp0.y );
	dsig[3] += DE[3][2] * ( deps.xy - depsvp0.xy );
	dsig[3] += DE[3][3] * ( deps.z  - depsvp0.z );

	//saving dsig in global variable dSTRESS
	for( row=0L; row<4L; row++ ) dSTRESS[row] = dsig[row];
		
	//total elastic stress
	sig[0] = sig0.x  + dsig[0];
	sig[1] = sig0.y  + dsig[1];
	sig[2] = sig0.xy + dsig[2];
	sig[3] = sig0.z  + dsig[3];

	//yield criterion
	Fcrit = Fmc_MMC( sig, Fi_MMC( Edp, mprs ), c_MMC( Edp, mprs ));

	if( Fcrit >= Zero )
	{
		//rearranging vectors
				
		//DFDEDQ and A
		DEP_MMC( sig, Edp, mprs);
					
		//Mohr-Coulomb time increment
		ctrl = sin( deg2rad( Fi_MMC( Edp, mprs ) )) * sin( deg2rad( Fi_MMC( Edp, mprs ))); 
		dt1 = Four * ( One + mprs.ni ) * ( One - Two * mprs.ni ) / ( mprs.E * ( One - Two * mprs.ni + ctrl ));
				
		//Potts time increment
		dt2 = One;
		if( FastDT == 0L )
			dt2 /= DFDEDQ + A;
		
		//finally
		if( dt1 < dt2 ) dt = dt1; else dt = dt2;

		if( dt < DT ) DT = dt;
	}
	else
	{
		for( row=0L; row<4L; row++ ) DQ[row] = Zero;
	}

	return Fcrit;

}//calcVP
//-------------------------------------------------------------------------------------


/* END */

#endif