/*
	File			:	IOTools.CPP
	Identification	:	Library of functions and procedures for FEM
						I/O routines : Saving data & results to disk files
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	30.06.2006
	Last modified	:	09.07.2006
	Content			:	

	size_t Read_BMP( const char *name )
	size_t Read_INP( const char *name )
	unsigned char Give_col( double x, double y )
	size_t Read_MSH( const char *name )
	void Save_Km( const char *name )
	void Save_vec( const char *name, double vec[MaxDim] )
	void Save_pp( const char *name )
	void Save_eps( const char *name )
	void Save_sig( const char *name )
	void Read_sig( const char *name )
	void Read_disp( const char *name )
	void Save_all( const char *name )
	void Save_par( const char *name )
	void CreateSign( const char *name )
	void Save4GID7( const char *name, int opt )
	void Save4SurferMatlab( const char *name )
*/

#ifndef _srokosz_IOTools_cpp

#define _srokosz_IOTools_cpp
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "FEMVars.h"
#include "FEMLib.h"
#include "MCPL.h"
#include "TQLE.h"
//elements
#include "T3.H"
#include "T6.H"
#include "Q4.H"
#include "Q8.H"
#include "Q9.H"

//-------------------------------------------------------------------------------------
/*  results :
				0 - ok
				1 - cannot open BMP
				2 - cannot read header
				3 - incorrect header (no "BM" at startup)
				4 - X-size incorrect
				5 - Y-size incorrect
				6 - number of colors incorrect
				7 - incorrect header length
				8 - incorrect data in header (bytes/pixel)
				9 - incorrect data in header (colors/pixel)
				10 - error in palette reading
				11 - error in data reading
*/
size_t Read_BMP( const char *name )
{
	FILE *BMP, *stream;
    unsigned long mx, my, datasize, i;
	size_t ctrl;
	unsigned short int BMPheader[27];
	struct RGB
	{
		unsigned char blue, green, red, cor;
	} palette[256];

	//opens BMP
	if(( BMP = fopen( name, "rb" )) == NULL )
	{
		return 1;
	}
	//reads header data: 54 bytes
	ctrl = fread( BMPheader, sizeof( unsigned short int ), 27, BMP );
	if( ctrl != 27 )
	{
		fclose( BMP );
		return 2;
	}
	
	stream = fopen( "BMPctrl.txt", "wt" );
	for( i=0; i<10; i++ ) fprintf( stream, "header[0%d]:  %d\n", i, BMPheader[i] );
	for( i=10; i<27; i++ ) fprintf( stream, "header[%d]:  %d\n", i, BMPheader[i] );
	fclose( stream );

	//header analysis
	if( BMPheader[0] != (unsigned short int) 19778 )
	{
		fclose( BMP );
		return 3;
	}
	mx = BMPheader[9];
	my = BMPheader[11];
	if(( mx - 8 * ( trunc( mx / 8 ))) != 0 )
	{
		fclose( BMP );
		return 4;
	}
	if(( my - 8 * ( trunc( my / 8 ))) != 0 )
	{
		fclose( BMP );
		return 5;
	}
	if( BMPheader[14] != 8 )
	{
		fclose( BMP );
		return 6;
	}
	if( BMPheader[5] != 1078 )
	{
		fclose( BMP );
		return 7;
	}
	if( BMPheader[13] != 1 )
	{
		fclose( BMP );
		return 8;
	}
	if( BMPheader[23] != 256 )
	{
		fclose( BMP );
		return 9;
	}
	if( BMPheader[25] != 256 )
	{
		fclose( BMP );
		return 9;
	}
	//pallette
	ctrl = fread( palette, sizeof( RGB ), 256, BMP );
	if( ctrl != 256 )
	{
		fclose( BMP );
		return 10;
	}
	//data
    datasize = BMPheader[17] + BMPheader[18] * (long) 65536;
	ctrl = fread( BMPData, sizeof( char ), datasize, BMP );
	fclose( BMP );
	if( ctrl != datasize )
	{
		return 11;
	}
	//all ok (till now)
	BMPheight = my;
	BMPwidth = mx;
	return 0;

}//Read_BMP
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//  results : number of material sets read
size_t Read_INP( const char *name )
{
	FILE *INP;
    long i, col, NPAR;
	char Info[100];
	double ro, E, ni, Ke, Edp_Fip1, Edp_cp1, Edp_Fip2, Edp_cp2, Edp_Fir, Edp_cr;
	double Fii, Fip, Fir, ci, cp, cr, beta, Psir;

	//zeroing
	for( i=0L; i<256L; i++ ) PAR[i] = MATPAR_ZERO;
	//opens INP
	if(( INP = fopen( name, "rt" )) == NULL )
	{
		return 1;
	}
	//reads version
	fscanf( INP, "%s\n", Info );
	if( strcmp( Info, (const char *)"FT09" ) != 0 ) 
	{
		fclose( INP );
		return 0;
	}
	//reads bmp data
	fscanf( INP, "%s %s\n", Info, Info ); 
	fscanf( INP, "%s %s\n", Info, Info ); 
	fscanf( INP, "%ld %ld\n", &NPAR, &MatIdentType );
	fscanf( INP, "%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s\n", Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info, Info );
	if( NPAR > 0L )
	{
		for( i=0L; i<NPAR; i++ )
		{
			fscanf( INP, "%ld %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE\n", &col, &ro, &E, &ni, &Ke, &Fii, &ci, &Edp_Fip1, &Edp_Fip2, &Fip, &Edp_cp1, &Edp_cp2, &cp, &Edp_Fir, &Fir, &Edp_cr, &cr, &beta, &Psir );
			PAR[col].ro = ro;
			PAR[col].E = E * (double) 1000.0;
			PAR[col].ni = ni;
			PAR[col].Ke = Ke;
			PAR[col].Fii = Fii;
			PAR[col].ci = ci;
			PAR[col].Edp_Fip1 = Edp_Fip1 / (double) 100.0;
			PAR[col].Edp_Fip2 = Edp_Fip2 / (double) 100.0;
			PAR[col].Fip = Fip;
			PAR[col].Edp_cp1 = Edp_cp1 / (double) 100.0;
			PAR[col].Edp_cp2 = Edp_cp2 / (double) 100.0;
			PAR[col].cp = cp;
			PAR[col].Edp_Fir = Edp_Fir / (double) 100.0;
			PAR[col].Fir = Fir;
			PAR[col].Edp_cr = Edp_cr / (double) 100.0;
			PAR[col].cr = cr;
			PAR[col].beta = beta;
			PAR[col].Psir = Psir;
		}
	}	
	fscanf( INP, "%s %s\n", Info, Info ); 
	fscanf( INP, "%s %s %s %s\n", Info, Info, Info, Info ); 
	fscanf( INP, "%lE %lE %ld %lE\n", &delta, &INCmax, &INCauto, &Fsafety );
	fscanf( INP, "%s %s\n", Info, Info ); 
	fscanf( INP, "%s %s %s %s %s %s %s\n", Info, Info, Info, Info, Info, Info, Info ); 
	fscanf( INP, "%lE %lE %lE %d %lE %d %lE\n", &Vcrit, &Fmin, &SStol, &SSmax, &NRtol, &NRmax, &NRRelax );
	fscanf( INP, "%s %s %s %s\n", Info, Info, Info, Info ); 
	fscanf( INP, "%lE %ld %ld %ld\n", &Collapse, &DEPuse, &VPuse, &FastDT );
	fscanf( INP, "%s %s\n", Info, Info ); 
	fscanf( INP, "%s %s\n", Info, Info );
	fscanf( INP, "%ld", &Nmonitor );
	for( i=0L; i<Nmonitor; i++ ) fscanf( INP, "%ld", &NODEmonitor[i] );
	fclose( INP );
	return NPAR;

}//Read_INP
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//  result : colour number at x:y position in BMP vector
unsigned char Give_col( double x, double y )
{
	long i, xi, yi;

	xi = (long) trunc( x * (double) 100.0 );
	yi = (long) trunc( y * (double) 100.0 );
	i = xi + ( yi * BMPwidth );
	if( i > ( BMPwidth * BMPheight - 1L ))
		return 0;
	else
		return BMPData[i];

}//Give_col
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//  results : 0 = ok
size_t Read_MSH( const char *name )
{
	FILE *MSH;
    long n1, n2, n3, n4, n5, n6, n7, n8, n9;
	long i, iter, N, typ;
	double Nx[9], ctrl, x, y;
	unsigned char col[9];
	MATPAR mpars;
	COOR2 Cntr;


	if(( MSH = fopen( name, "rt" )) == NULL )	return 1;

	//nodes
	fscanf( MSH, "%D\n", &N);
	if( N > MaxNode )
	{
		fclose( MSH );
		return 2;
	}
	for( i=0; i<N; i++ )
	{
		fscanf( MSH, "%lE %lE\n", &x, &y);
		mkNODE( i, x, y );
	}
	
	//elements
	totalGP = 0L;
	fscanf( MSH, "%D\n", &N );
	if( N > MaxElem )
	{
		fclose( MSH );
		return 3;
	}
	for( i=0; i<N; i++ )
	{
		//zeroing material properties
		mpars = MATPAR_ZERO;

		fscanf( MSH, "%D", &typ);
		if( typ == 3 ) 
		{
			fscanf( MSH, "%D %D %D\n", &n1, &n2, &n3);
			mkT3( i, n1, n2, n3 );
			//gravity loading
			col[0] = Give_col( NODE[n1].x, NODE[n1].y );
			Nx[0] = N3( i );
			uploadNODE( n1, Zero, g * PAR[col[0]].ro * Nx[0] );
			col[1] = Give_col( NODE[n2].x, NODE[n2].y );
			Nx[1] = Nx[0];
			uploadNODE( n2, Zero, g * PAR[col[1]].ro * Nx[1] );
			col[2] = Give_col( NODE[n3].x, NODE[n3].y );
			Nx[2] = Nx[1];
			uploadNODE( n3, Zero, g * PAR[col[2]].ro * Nx[2] );			
			if( MatIdentType == 1L )
			{
				//average material parameters
				ctrl = Nx[0] + Nx[1] + Nx[2];
				for( iter=0L; iter<3L; iter++ )
				{
					mpars.Ke += Nx[iter] / ctrl * PAR[col[iter]].Ke;
					mpars.E += Nx[iter] / ctrl * PAR[col[iter]].E;
					mpars.ni += Nx[iter] / ctrl * PAR[col[iter]].ni;
					mpars.Fii += Nx[iter] / ctrl * PAR[col[iter]].Fii;
					mpars.ci += Nx[iter] / ctrl * PAR[col[iter]].ci;
					mpars.Fip += Nx[iter] / ctrl * PAR[col[iter]].Fip;
					mpars.cp += Nx[iter] / ctrl * PAR[col[iter]].cp;
					mpars.Fir += Nx[iter] / ctrl * PAR[col[iter]].Fir;
					mpars.cr += Nx[iter] / ctrl * PAR[col[iter]].cr;
					mpars.Edp_Fip1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip1;
					mpars.Edp_Fip2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip2;
					mpars.Edp_Fir += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fir;
					mpars.Edp_cp1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp1;
					mpars.Edp_cp2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp2;
					mpars.Edp_cr += Nx[iter] / ctrl * PAR[col[iter]].Edp_cr;
					mpars.beta += Nx[iter] / ctrl * PAR[col[iter]].beta;
					mpars.Psir += Nx[iter] / ctrl * PAR[col[iter]].Psir;
				}
			}
			else
			{
				//material parameters for center point
				Cntr = CenterT3( i );
				col[0] = Give_col( Cntr.x, Cntr.y );
				mpars.Ke = PAR[col[0]].Ke;
				mpars.E = PAR[col[0]].E;
				mpars.ni = PAR[col[0]].ni;
				mpars.Fii = PAR[col[0]].Fii;
				mpars.ci = PAR[col[0]].ci;
				mpars.Fip = PAR[col[0]].Fip;
				mpars.cp = PAR[col[0]].cp;
				mpars.Fir = PAR[col[0]].Fir;
				mpars.cr = PAR[col[0]].cr;
				mpars.Edp_Fip1 = PAR[col[0]].Edp_Fip1;
				mpars.Edp_Fip2 = PAR[col[0]].Edp_Fip2;
				mpars.Edp_Fir = PAR[col[0]].Edp_Fir;
				mpars.Edp_cp1 = PAR[col[0]].Edp_cp1;
				mpars.Edp_cp2 = PAR[col[0]].Edp_cp2;
				mpars.Edp_cr = PAR[col[0]].Edp_cr;
				mpars.beta = PAR[col[0]].beta;
				mpars.Psir = PAR[col[0]].Psir;
			}
			totalGP += 1L;
		}
		if( typ == 4 ) 
		{
			fscanf( MSH, "%D %D %D %D\n", &n1, &n2, &n3, &n4);
			mkQ4( i, n1, n2, n3, n4 );
			//gravity loading
			col[0] = Give_col( NODE[n1].x, NODE[n1].y );
			Nx[0] = N4( i, 1 );
			uploadNODE( n1, Zero, g * PAR[col[0]].ro * Nx[0] );
			col[1] = Give_col( NODE[n2].x, NODE[n2].y );
			Nx[1] = N4( i, 2 );
			uploadNODE( n2, Zero, g * PAR[col[1]].ro * Nx[1] );
			col[2] = Give_col( NODE[n3].x, NODE[n3].y );
			Nx[2] = N4( i, 3 );
			uploadNODE( n3, Zero, g * PAR[col[2]].ro * Nx[2] );
			col[3] = Give_col( NODE[n4].x, NODE[n4].y );
			Nx[3] = N4( i, 4 );
			uploadNODE( n4, Zero, g * PAR[col[3]].ro * Nx[3] );
			if( MatIdentType == 1L )
			{
				//average material parameters
				ctrl = Nx[0] + Nx[1] + Nx[2] + Nx[3];
				for( iter=0L; iter<4L; iter++ )
				{
					mpars.Ke += Nx[iter] / ctrl * PAR[col[iter]].Ke;
					mpars.E += Nx[iter] / ctrl * PAR[col[iter]].E;
					mpars.ni += Nx[iter] / ctrl * PAR[col[iter]].ni;
					mpars.Fii += Nx[iter] / ctrl * PAR[col[iter]].Fii;
					mpars.ci += Nx[iter] / ctrl * PAR[col[iter]].ci;
					mpars.Fip += Nx[iter] / ctrl * PAR[col[iter]].Fip;
					mpars.cp += Nx[iter] / ctrl * PAR[col[iter]].cp;
					mpars.Fir += Nx[iter] / ctrl * PAR[col[iter]].Fir;
					mpars.cr += Nx[iter] / ctrl * PAR[col[iter]].cr;
					mpars.Edp_Fip1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip1;
					mpars.Edp_Fip2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip2;
					mpars.Edp_Fir += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fir;
					mpars.Edp_cp1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp1;
					mpars.Edp_cp2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp2;
					mpars.Edp_cr += Nx[iter] / ctrl * PAR[col[iter]].Edp_cr;
					mpars.beta += Nx[iter] / ctrl * PAR[col[iter]].beta;
					mpars.Psir += Nx[iter] / ctrl * PAR[col[iter]].Psir;
				}
			}
			else
			{
				//material parameters for center point
				Cntr = CenterQ4( i );
				col[0] = Give_col( Cntr.x, Cntr.y );
				mpars.Ke = PAR[col[0]].Ke;
				mpars.E = PAR[col[0]].E;
				mpars.ni = PAR[col[0]].ni;
				mpars.Fii = PAR[col[0]].Fii;
				mpars.ci = PAR[col[0]].ci;
				mpars.Fip = PAR[col[0]].Fip;
				mpars.cp = PAR[col[0]].cp;
				mpars.Fir = PAR[col[0]].Fir;
				mpars.cr = PAR[col[0]].cr;
				mpars.Edp_Fip1 = PAR[col[0]].Edp_Fip1;
				mpars.Edp_Fip2 = PAR[col[0]].Edp_Fip2;
				mpars.Edp_Fir = PAR[col[0]].Edp_Fir;
				mpars.Edp_cp1 = PAR[col[0]].Edp_cp1;
				mpars.Edp_cp2 = PAR[col[0]].Edp_cp2;
				mpars.Edp_cr = PAR[col[0]].Edp_cr;
				mpars.beta = PAR[col[0]].beta;
				mpars.Psir = PAR[col[0]].Psir;
			}
			totalGP += 4L;
		}
		if( typ == 6 ) 
		{
			fscanf( MSH, "%D %D %D %D %D %D\n", &n1, &n2, &n3, &n4, &n5, &n6);
			mkT6( i, n1, n2, n3, n4, n5, n6 );
			//gravity loading
			col[0] = Give_col( NODE[n1].x, NODE[n1].y );
			Nx[0] = N6( i, 1 );
			uploadNODE( n1, Zero, g * PAR[col[0]].ro * Nx[0] );
			col[1] = Give_col( NODE[n2].x, NODE[n2].y );
			Nx[1] = N6( i, 2 );
			uploadNODE( n2, Zero, g * PAR[col[1]].ro * Nx[1] );
			col[2] = Give_col( NODE[n3].x, NODE[n3].y );
			Nx[2] = N6( i, 3 );
			uploadNODE( n3, Zero, g * PAR[col[2]].ro * Nx[2] );
			col[3] = Give_col( NODE[n4].x, NODE[n4].y );
			Nx[3] = N6( i, 4 );
			uploadNODE( n4, Zero, g * PAR[col[3]].ro * Nx[3] );
			col[4] = Give_col( NODE[n5].x, NODE[n5].y );
			Nx[4] = N6( i, 5 );
			uploadNODE( n5, Zero, g * PAR[col[4]].ro * Nx[4] );
			col[5] = Give_col( NODE[n6].x, NODE[n6].y );
			Nx[5] = N6( i, 6 );
			uploadNODE( n6, Zero, g * PAR[col[5]].ro * Nx[5] );
			if( MatIdentType == 1L )
			{
				//average material parameters
				ctrl = Nx[0] + Nx[1] + Nx[2] + Nx[3] + Nx[4] + Nx[5];
				for( iter=0L; iter<6L; iter++ )
				{
					mpars.Ke += Nx[iter] / ctrl * PAR[col[iter]].Ke;
					mpars.E += Nx[iter] / ctrl * PAR[col[iter]].E;
					mpars.ni += Nx[iter] / ctrl * PAR[col[iter]].ni;
					mpars.Fii += Nx[iter] / ctrl * PAR[col[iter]].Fii;
					mpars.ci += Nx[iter] / ctrl * PAR[col[iter]].ci;
					mpars.Fip += Nx[iter] / ctrl * PAR[col[iter]].Fip;
					mpars.cp += Nx[iter] / ctrl * PAR[col[iter]].cp;
					mpars.Fir += Nx[iter] / ctrl * PAR[col[iter]].Fir;
					mpars.cr += Nx[iter] / ctrl * PAR[col[iter]].cr;
					mpars.Edp_Fip1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip1;
					mpars.Edp_Fip2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip2;
					mpars.Edp_Fir += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fir;
					mpars.Edp_cp1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp1;
					mpars.Edp_cp2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp2;
					mpars.Edp_cr += Nx[iter] / ctrl * PAR[col[iter]].Edp_cr;
					mpars.beta += Nx[iter] / ctrl * PAR[col[iter]].beta;
					mpars.Psir += Nx[iter] / ctrl * PAR[col[iter]].Psir;
				}
			}
			else
			{
				//material parameters for center point
				Cntr = CenterT6( i );
				col[0] = Give_col( Cntr.x, Cntr.y );
				mpars.Ke = PAR[col[0]].Ke;
				mpars.E = PAR[col[0]].E;
				mpars.ni = PAR[col[0]].ni;
				mpars.Fii = PAR[col[0]].Fii;
				mpars.ci = PAR[col[0]].ci;
				mpars.Fip = PAR[col[0]].Fip;
				mpars.cp = PAR[col[0]].cp;
				mpars.Fir = PAR[col[0]].Fir;
				mpars.cr = PAR[col[0]].cr;
				mpars.Edp_Fip1 = PAR[col[0]].Edp_Fip1;
				mpars.Edp_Fip2 = PAR[col[0]].Edp_Fip2;
				mpars.Edp_Fir = PAR[col[0]].Edp_Fir;
				mpars.Edp_cp1 = PAR[col[0]].Edp_cp1;
				mpars.Edp_cp2 = PAR[col[0]].Edp_cp2;
				mpars.Edp_cr = PAR[col[0]].Edp_cr;
				mpars.beta = PAR[col[0]].beta;
				mpars.Psir = PAR[col[0]].Psir;
			}
			totalGP += 3L;
		}
		if( typ == 8 ) 
		{
			fscanf( MSH, "%D %D %D %D %D %D %D %D\n", &n1, &n2, &n3, &n4, &n5, &n6, &n7, &n8);
			mkQ8( i, n1, n2, n3, n4, n5, n6, n7, n8 );
			//gravity loading
			col[0] = Give_col( NODE[n1].x, NODE[n1].y );
			Nx[0] = N8( i, 1 );
			uploadNODE( n1, Zero, g * PAR[col[0]].ro * Nx[0] );
			col[1] = Give_col( NODE[n2].x, NODE[n2].y );
			Nx[1] = N8( i, 2 );
			uploadNODE( n2, Zero, g * PAR[col[1]].ro * Nx[1] );
			col[2] = Give_col( NODE[n3].x, NODE[n3].y );
			Nx[2] = N8( i, 3 );
			uploadNODE( n3, Zero, g * PAR[col[2]].ro * Nx[2] );
			col[3] = Give_col( NODE[n4].x, NODE[n4].y );
			Nx[3] = N8( i, 4 );
			uploadNODE( n4, Zero, g * PAR[col[3]].ro * Nx[3] );
			col[4] = Give_col( NODE[n5].x, NODE[n5].y );
			Nx[4] = N8( i, 5 );
			uploadNODE( n5, Zero, g * PAR[col[4]].ro * Nx[4] );
			col[5] = Give_col( NODE[n6].x, NODE[n6].y );
			Nx[5] = N8( i, 6 );
			uploadNODE( n6, Zero, g * PAR[col[5]].ro * Nx[5] );
			col[6] = Give_col( NODE[n7].x, NODE[n7].y );
			Nx[6] = N8( i, 7 );
			uploadNODE( n7, Zero, g * PAR[col[6]].ro * Nx[6] );
			col[7] = Give_col( NODE[n8].x, NODE[n8].y );
			Nx[7] = N8( i, 8 );
			uploadNODE( n8, Zero, g * PAR[col[7]].ro * Nx[7] );
			if( MatIdentType == 1L )
			{
				//average material parameters
				ctrl = Nx[0] + Nx[1] + Nx[2] + Nx[3] + Nx[4] + Nx[5] + Nx[6] + Nx[7];
				for( iter=0L; iter<8L; iter++ )
				{
					mpars.Ke += Nx[iter] / ctrl * PAR[col[iter]].Ke;
					mpars.E += Nx[iter] / ctrl * PAR[col[iter]].E;
					mpars.ni += Nx[iter] / ctrl * PAR[col[iter]].ni;
					mpars.Fii += Nx[iter] / ctrl * PAR[col[iter]].Fii;
					mpars.ci += Nx[iter] / ctrl * PAR[col[iter]].ci;
					mpars.Fip += Nx[iter] / ctrl * PAR[col[iter]].Fip;
					mpars.cp += Nx[iter] / ctrl * PAR[col[iter]].cp;
					mpars.Fir += Nx[iter] / ctrl * PAR[col[iter]].Fir;
					mpars.cr += Nx[iter] / ctrl * PAR[col[iter]].cr;
					mpars.Edp_Fip1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip1;
					mpars.Edp_Fip2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip2;
					mpars.Edp_Fir += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fir;
					mpars.Edp_cp1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp1;
					mpars.Edp_cp2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp2;
					mpars.Edp_cr += Nx[iter] / ctrl * PAR[col[iter]].Edp_cr;
					mpars.beta += Nx[iter] / ctrl * PAR[col[iter]].beta;
					mpars.Psir += Nx[iter] / ctrl * PAR[col[iter]].Psir;
				}
			}
			else
			{
				//material parameters for center point
				Cntr = CenterQ8( i );
				col[0] = Give_col( Cntr.x, Cntr.y );
				mpars.Ke = PAR[col[0]].Ke;
				mpars.E = PAR[col[0]].E;
				mpars.ni = PAR[col[0]].ni;
				mpars.Fii = PAR[col[0]].Fii;
				mpars.ci = PAR[col[0]].ci;
				mpars.Fip = PAR[col[0]].Fip;
				mpars.cp = PAR[col[0]].cp;
				mpars.Fir = PAR[col[0]].Fir;
				mpars.cr = PAR[col[0]].cr;
				mpars.Edp_Fip1 = PAR[col[0]].Edp_Fip1;
				mpars.Edp_Fip2 = PAR[col[0]].Edp_Fip2;
				mpars.Edp_Fir = PAR[col[0]].Edp_Fir;
				mpars.Edp_cp1 = PAR[col[0]].Edp_cp1;
				mpars.Edp_cp2 = PAR[col[0]].Edp_cp2;
				mpars.Edp_cr = PAR[col[0]].Edp_cr;
				mpars.beta = PAR[col[0]].beta;
				mpars.Psir = PAR[col[0]].Psir;
			}
			totalGP += 9L;
		}
		if( typ == 9 ) 
		{
			fscanf( MSH, "%D %D %D %D %D %D %D %D %D\n", &n1, &n2, &n3, &n4, &n5, &n6, &n7, &n8, &n9);
			mkQ9( i, n1, n2, n3, n4, n5, n6, n7, n8, n9 );
			//gravity loading
			col[0] = Give_col( NODE[n1].x, NODE[n1].y );
			Nx[0] = N9( i, 1 );
			uploadNODE( n1, Zero, g * PAR[col[0]].ro * Nx[0] );
			col[1] = Give_col( NODE[n2].x, NODE[n2].y );
			Nx[1] = N9( i, 2 );
			uploadNODE( n2, Zero, g * PAR[col[1]].ro * Nx[1] );
			col[2] = Give_col( NODE[n3].x, NODE[n3].y );
			Nx[2] = N9( i, 3 );
			uploadNODE( n3, Zero, g * PAR[col[2]].ro * Nx[2] );
			col[3] = Give_col( NODE[n4].x, NODE[n4].y );
			Nx[3] = N9( i, 4 );
			uploadNODE( n4, Zero, g * PAR[col[3]].ro * Nx[3] );
			col[4] = Give_col( NODE[n5].x, NODE[n5].y );
			Nx[4] = N9( i, 5 );
			uploadNODE( n5, Zero, g * PAR[col[4]].ro * Nx[4] );
			col[5] = Give_col( NODE[n6].x, NODE[n6].y );
			Nx[5] = N9( i, 6 );
			uploadNODE( n6, Zero, g * PAR[col[5]].ro * Nx[5] );
			col[6] = Give_col( NODE[n7].x, NODE[n7].y );
			Nx[6] = N9( i, 7 );
			uploadNODE( n7, Zero, g * PAR[col[6]].ro * Nx[6] );
			col[7] = Give_col( NODE[n8].x, NODE[n8].y );
			Nx[7] = N9( i, 8 );
			uploadNODE( n8, Zero, g * PAR[col[7]].ro * Nx[7] );
			col[8] = Give_col( NODE[n9].x, NODE[n9].y );
			Nx[8] = N9( i, 9 );
			uploadNODE( n9, Zero, g * PAR[col[8]].ro * Nx[8] );
			if( MatIdentType == 1L )
			{
				//average material parameters
				ctrl = Nx[0] + Nx[1] + Nx[2] + Nx[3] + Nx[4] + Nx[5] + Nx[6] + Nx[7] + Nx[8];
				for( iter=0L; iter<9L; iter++ )
				{
					mpars.Ke += Nx[iter] / ctrl * PAR[col[iter]].Ke;
					mpars.E += Nx[iter] / ctrl * PAR[col[iter]].E;
					mpars.ni += Nx[iter] / ctrl * PAR[col[iter]].ni;
					mpars.Fii += Nx[iter] / ctrl * PAR[col[iter]].Fii;
					mpars.ci += Nx[iter] / ctrl * PAR[col[iter]].ci;
					mpars.Fip += Nx[iter] / ctrl * PAR[col[iter]].Fip;
					mpars.cp += Nx[iter] / ctrl * PAR[col[iter]].cp;
					mpars.Fir += Nx[iter] / ctrl * PAR[col[iter]].Fir;
					mpars.cr += Nx[iter] / ctrl * PAR[col[iter]].cr;
					mpars.Edp_Fip1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip1;
					mpars.Edp_Fip2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fip2;
					mpars.Edp_Fir += Nx[iter] / ctrl * PAR[col[iter]].Edp_Fir;
					mpars.Edp_cp1 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp1;
					mpars.Edp_cp2 += Nx[iter] / ctrl * PAR[col[iter]].Edp_cp2;
					mpars.Edp_cr += Nx[iter] / ctrl * PAR[col[iter]].Edp_cr;
					mpars.beta += Nx[iter] / ctrl * PAR[col[iter]].beta;
					mpars.Psir += Nx[iter] / ctrl * PAR[col[iter]].Psir;
				}
			}
			else
			{
				//material parameters for center point
				Cntr = CenterQ9( i );
				col[0] = Give_col( Cntr.x, Cntr.y );
				mpars.Ke = PAR[col[0]].Ke;
				mpars.E = PAR[col[0]].E;
				mpars.ni = PAR[col[0]].ni;
				mpars.Fii = PAR[col[0]].Fii;
				mpars.ci = PAR[col[0]].ci;
				mpars.Fip = PAR[col[0]].Fip;
				mpars.cp = PAR[col[0]].cp;
				mpars.Fir = PAR[col[0]].Fir;
				mpars.cr = PAR[col[0]].cr;
				mpars.Edp_Fip1 = PAR[col[0]].Edp_Fip1;
				mpars.Edp_Fip2 = PAR[col[0]].Edp_Fip2;
				mpars.Edp_Fir = PAR[col[0]].Edp_Fir;
				mpars.Edp_cp1 = PAR[col[0]].Edp_cp1;
				mpars.Edp_cp2 = PAR[col[0]].Edp_cp2;
				mpars.Edp_cr = PAR[col[0]].Edp_cr;
				mpars.beta = PAR[col[0]].beta;
				mpars.Psir = PAR[col[0]].Psir;
			}
			totalGP += 9L;
		}

		//update element material properties
		mpars.Fii = rad2deg(atan(tan(deg2rad(mpars.Fii))/Fsafety));
		mpars.Fip = rad2deg(atan(tan(deg2rad(mpars.Fip))/Fsafety));
		mpars.Fir = rad2deg(atan(tan(deg2rad(mpars.Fir))/Fsafety));
		mpars.Psir = rad2deg(atan(tan(deg2rad(mpars.Psir))/Fsafety));
		mpars.ci /= Fsafety;
		mpars.cp /= Fsafety;
		mpars.cr /= Fsafety;
        ELEM[i].matpars = mpars;
	}

	//boundary conditions
	fscanf( MSH, "%D\n", &N );
	if( N > MaxBound )
	{
		fclose( MSH );
		return 4;
	}
	for( i=0; i<N; i++ )
	{
		fscanf( MSH, "%D %lE %lE\n", &n1, &x, &y);
		saveCONSTRAINT( n1, x, y );
	}	
	fclose( MSH );

	return 0;

}//Read_MSH
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_Km( const char *name )
{

	FILE *stream;
	long row, col;

	stream = fopen( name, "wt");
	for(row=0L; row<NDIM; row++)
	{
		fprintf(stream,"u%d   ",row);
		for(col=0L; col<NDIM; col++)
			fprintf(stream, "%12.3E  ", Km[row][col]);
		fprintf(stream, "\n");
	}
	fclose(stream);
}//Save_Km
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_vec( const char *name, double vec[MaxDim] )
{
	FILE *stream;
	long row;
	COOR2 z;

	stream = fopen( name, "wt");
	for(row=0; row<(NNODE); row++)
	{
		z = readNODE( vec, row );
		fprintf(stream, "%5d    %12.3E   %12.3E\n", row, z.x, z.y );
	}
	fclose(stream);
}//Save_vec
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_monitor( const char *name, long incr )
{
	FILE *stream;
	char incname[250], incnum[10];
	long i;
	COOR2 z;

	//saving nodes' results in separate files
	for( i=0L; i<Nmonitor; i++ )
	{		
		strcpy( incname, name );
		ltoa( NODEmonitor[i], incnum, 10 );
		strcat( incname, ".");
		strcat( incname, incnum );
		strcat( incname, ".TXT");
		stream = fopen( incname, "at");
		z = readNODE( u_act, NODEmonitor[i] );		
		fprintf(stream,"inc: %5d  u(act).x: %E  u(act).y: %E", incr, z.x, z.y);
		z = readNODE( u_incr, NODEmonitor[i] );		
		fprintf(stream,"  u(inc).x: %E  u(inc).y: %E", z.x, z.y);
		z = readNODE( u_elas, NODEmonitor[i] );		
		fprintf(stream,"  u(ela).x: %E  u(ela).y: %E", z.x, z.y);
		z = readNODE( fResidual, NODEmonitor[i] );		
		fprintf(stream,"  f(int).x: %E  f(int).y: %E", z.x, z.y);
		z = readNODE( f_ext, NODEmonitor[i] );		
		fprintf(stream,"  f(ext).x: %E  f(ext).y: %E", z.x, z.y);
		z = readNODE( f_total, NODEmonitor[i] );	
		fprintf(stream,"  f(tot).x: %E  f(tot).y: %E", z.x, z.y);
		z = readNODE( f, NODEmonitor[i] );		
		fprintf(stream,"  f(act).x: %E  f(act).y: %E\n", z.x, z.y);
		fclose(stream);
	}
	
	//saving nodes' results in one file for load-disp curve preparation
	strcpy( incname, name );
	strcat( incname, ".LOAD.DISP.TXT");
	stream = fopen( incname, "at");
	for( i=0L; i<Nmonitor; i++ )
	{		
		z = readNODE( u_act, NODEmonitor[i] );		
		fprintf(stream," inc:%4d u.x: %E u.y: %E", incr, z.x, z.y);
		z = readNODE( fResidual, NODEmonitor[i] );		
		fprintf(stream," f.x: %E f.y: %E", z.x, z.y);
	}
	fprintf(stream, "\n");
	fclose(stream);
}//Save_monitor
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_pp( const char *name )
{
	FILE *stream;
	long row, iElem, k;
	COOR2 z;
	double t;

	stream = fopen( name, "wt");
	for(iElem=0; iElem<NELEM; iElem++)
	{
		if( ELEM[iElem].typ == 3 ) row = 1;
		if( ELEM[iElem].typ == 4 ) row = 4;
		if( ELEM[iElem].typ == 6 ) row = 3;
		if( ELEM[iElem].typ == 8 ) row = 9;
		if( ELEM[iElem].typ == 9 ) row = 9;
		for(k=0; k<row; k++)
		{
			t = ELEM[iElem].u[k];
			z = ELEM[iElem].Gauss[k];
			fprintf(stream, "%5d    %5d     %12.3E    %12.3E    %12.3E\n", iElem, k, z.x, z.y, t );
		}
	}
	fclose(stream);
}//Save_pp
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_eps( const char *name )
{
	FILE *stream;
	long row, iElem, k;
	COOR2 z;
	COOR3 t;

	stream = fopen( name, "wt");
	for(iElem=0; iElem<NELEM; iElem++)
	{
		if( ELEM[iElem].typ == 3 ) row = 1;
		if( ELEM[iElem].typ == 4 ) row = 4;
		if( ELEM[iElem].typ == 6 ) row = 3;
		if( ELEM[iElem].typ == 8 ) row = 9;
		if( ELEM[iElem].typ == 9 ) row = 9;
		for(k=0; k<row; k++)
		{
			t = ELEM[iElem].eps[k];
			z = ELEM[iElem].Gauss[k];
			fprintf(stream, "%5d    %5d     %12.3E    %12.3E    %12.3E    %12.3E    %12.3E\n", iElem, k, z.x, z.y, t.x, t.y, t.xy );
		}
	}
	fclose(stream);
}//Save_eps
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_sig( const char *name )
{
	FILE *stream;
	long row, iElem, k;
	COOR2 z;
	COOR3 t;

	stream = fopen( name, "wt");
	for(iElem=0; iElem<NELEM; iElem++)
	{
		if( ELEM[iElem].typ == 3 ) row = 1;
		if( ELEM[iElem].typ == 4 ) row = 4;
		if( ELEM[iElem].typ == 6 ) row = 3;
		if( ELEM[iElem].typ == 8 ) row = 9;
		if( ELEM[iElem].typ == 9 ) row = 9;
		for(k=0; k<row; k++)
		{
			t = ELEM[iElem].sig[k];
			z = ELEM[iElem].Gauss[k];
			fprintf(stream, "%5d    %5d     %12.3E    %12.3E    %12.3E    %12.3E    %12.3E    %12.3E\n", iElem, k, z.x, z.y, t.x, t.y, t.xy, t.z );
		}
	}
	fclose(stream);
}//Save_sig
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Read_sig( const char *name )
{
	FILE *stream;
	long row, iElem, k, ae, bg;
	double sx, sy, txy, sz, xgp, ygp;
	COOR2 z;
	COOR3 t;

	stream = fopen( name, "rt");
	for(iElem=0; iElem<NELEM; iElem++)
	{
		if( ELEM[iElem].typ == 3 ) row = 1;
		if( ELEM[iElem].typ == 4 ) row = 4;
		if( ELEM[iElem].typ == 6 ) row = 3;
		if( ELEM[iElem].typ == 8 ) row = 9;
		if( ELEM[iElem].typ == 9 ) row = 9;
		for(k=0L; k<row; k++)
		{
			fscanf(stream, "%ld %ld %lf %lf %lf %lf %lf %lf", &ae, &bg, &xgp, &ygp, &sx, &sy, &txy, &sz );
			t.x = sx;
			t.xy = txy;
			t.y = sy;
			t.z = sz;
			z.x = xgp;
			z.y = ygp;
			ELEM[iElem].sig[k] = t;
			//ELEM[iElem].Gauss[k] = z;
		}
	}
	fclose(stream);
}//Read_sig
//-------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------
void Read_loads( const char *name )
{
	FILE *stream;
	long iN, node, maxN;
	double Fx, Fy;

	stream = fopen( name, "rt");
	fscanf(stream, "%ld", &maxN );
	if( maxN > 0L )
	{
		for(iN=0L; iN<maxN; iN++)
		{
			fscanf(stream, "%ld %lf %lf", &node, &Fx, &Fy );
			uploadNODE( node, Fx, Fy );
		}
	}
	fclose(stream);
}//Read_loads
//-------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------
void Read_disp( const char *name )
{
	FILE *stream;
	long iN, node, maxN;
	double Ux, Uy;

	stream = fopen( name, "rt");
	fscanf(stream, "%ld", &maxN );	
	for(iN=0L; iN<maxN; iN++)
	if( maxN > 0L )
	{
		{
			fscanf(stream, "%ld %lf %lf", &node, &Ux, &Uy );
			saveCONSTRAINT( node, Ux, Uy );
		}
	}
	fclose(stream);
}//Read_disp
//-------------------------------------------------------------------------------------


//-------------------------------------------------------------------------------------
void Save_all( const char *name )
{
	FILE *stream;
	long row, k, i;
	COOR2 z;
	COOR3 t;

	stream = fopen( name, "wt");
    fprintf(stream,"%d  %E\n",NNODE, Fmin); //liczba wezlow, min F
    
	for( i=0; i<NNODE; i++)
	{
		z = readNODE( u_act, i );		
		fprintf(stream,"%d  %E  %E  %E  %E  ", i+1, NODE[i].x, NODE[i].y, z.x, z.y);
		z = readNODE( u_incr, i );		
		fprintf(stream,"%E  %E  ", z.x, z.y);
		z = readNODE( u_elas, i );		
		fprintf(stream,"%E  %E\n", z.x, z.y);
	}

    fprintf(stream,"%d\n",NELEM); //liczba elementow
    
	for( i=0; i<NELEM; i++ )
	{
		fprintf(stream,"%d  %d  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E  %E\n",
			i+1 ,ELEM[i].typ, ELEM[i].matpars.E, ELEM[i].matpars.ni, ELEM[i].matpars.Ke, ELEM[i].matpars.Fii, ELEM[i].matpars.ci, ELEM[i].matpars.Edp_Fip1,
			ELEM[i].matpars.Edp_Fip2, ELEM[i].matpars.Fip, ELEM[i].matpars.Edp_cp1, ELEM[i].matpars.Edp_cp2, ELEM[i].matpars.cp, 
			ELEM[i].matpars.Edp_Fir, ELEM[i].matpars.Fir, ELEM[i].matpars.Edp_cr, ELEM[i].matpars.cr, ELEM[i].matpars.beta, ELEM[i].matpars.Psir );
		if( ELEM[i].typ == 3 )
		{
			row = 1;
			fprintf(stream, "%d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1);
		}
		if( ELEM[i].typ == 4 )
		{
			row = 4;
			fprintf(stream, "%d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1);
		}
		if( ELEM[i].typ == 6 )
		{
			row = 3;
			fprintf(stream, "%d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
														ELEM[i].node[4]+1, ELEM[i].node[5]+1);
		}
		if( ELEM[i].typ == 8 )
		{
			row = 9;
			fprintf(stream, "%d  %d  %d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
							ELEM[i].node[4]+1, ELEM[i].node[5]+1, ELEM[i].node[6]+1, ELEM[i].node[7]+1);
		}
		if( ELEM[i].typ == 9 )
		{
			row = 9;
			fprintf(stream, "%d  %d  %d  %d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
							ELEM[i].node[4]+1, ELEM[i].node[5]+1, ELEM[i].node[6]+1, ELEM[i].node[7]+1, ELEM[i].node[8]+1);
		}
		for(k=0; k<row; k++)
		{
			z = ELEM[i].Gauss[k];
			fprintf(stream, "%E  %E  %d\n", z.x, z.y, ELEM[i].plas[k]);
			t = ELEM[i].eps[k];
			fprintf(stream, "%E  %E  %E  ", t.x, t.y, t.xy );
			t = ELEM[i].deps[k];
			fprintf(stream, "%E  %E  %E  ", t.x, t.y, t.xy );
			t = ELEM[i].depsvp[k];
			fprintf(stream, "%E  %E  %E\n", t.x, t.y, t.xy );
			t = ELEM[i].sig[k];
			fprintf(stream, "%E  %E  %E  %E  %E  ", t.x, t.y, t.xy, t.z, ELEM[i].Fmc[k] );
			t = ELEM[i].dsig[k];
			fprintf(stream, "%E  %E  %E  %E  ", t.x, t.y, t.xy, t.z );
			t = ELEM[i].dsigvp[k];
			fprintf(stream, "%E  %E  %E  %E\n", t.x, t.y, t.xy, t.z );
			fprintf(stream, "%E  %E\n", ELEM[i].u[k], ELEM[i].Edp[k] );
		}
	}
	fclose(stream);
}//Save_all
//-------------------------------------------------------------------------------------

//saving data for postprocessing in Matlab or Surfer
//-------------------------------------------------------------------------------------
void Save4SurferMatlab( const char *name )
{
	FILE *stream;
	long i;
	COOR2 z;

	stream = fopen( name, "wt");
    fprintf(stream,"x  y  ux  uy  u\n",NNODE); 
    
	for( i=0; i<NNODE; i++)
	{
		z = readNODE( u_act, i );		
		fprintf(stream,"%E  %E  %E  %E  %E\n", NODE[i].x, NODE[i].y, z.x, z.y, hypot( z.x, z.y ));
	}

	fclose(stream);
}//Save4SurferMatlab
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void Save_par( const char *name )
{
	FILE *stream;
	long row, i;

	stream = fopen( name, "wt");
    
	for( i=0L; i<256L; i++ )
			fprintf( stream, "%d %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE %lE\n", i, PAR[i].ro, PAR[i].E, PAR[i].ni, PAR[i].Ke, PAR[i].Fii, PAR[i].ci, PAR[i].Edp_Fip1, PAR[i].Edp_Fip2, PAR[i].Fip, PAR[i].Edp_cp1, PAR[i].Edp_cp2, PAR[i].cp, PAR[i].Edp_Fir, PAR[i].Fir, PAR[i].Edp_cr, PAR[i].cr, PAR[i].beta, PAR[i].Psir );
    
    fprintf(stream,"%d\n",NELEM); //liczba elementow
    
	for( i=0; i<NELEM; i++ )
	{
		fprintf(stream,"%d  %d\n",i+1 ,ELEM[i].typ);
		if( ELEM[i].typ == 3 )
		{
			row = 1;
			fprintf(stream, "%d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1);
		}
		if( ELEM[i].typ == 4 )
		{
			row = 4;
			fprintf(stream, "%d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1);
		}
		if( ELEM[i].typ == 6 )
		{
			row = 3;
			fprintf(stream, "%d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
														ELEM[i].node[4]+1, ELEM[i].node[5]+1);
		}
		if( ELEM[i].typ == 8 )
		{
			row = 9;
			fprintf(stream, "%d  %d  %d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
							ELEM[i].node[4]+1, ELEM[i].node[5]+1, ELEM[i].node[6]+1, ELEM[i].node[7]+1);
		}
		if( ELEM[i].typ == 9 )
		{
			row = 9;
			fprintf(stream, "%d  %d  %d  %d  %d  %d  %d  %d  %d\n", ELEM[i].node[0]+1, ELEM[i].node[1]+1, ELEM[i].node[2]+1, ELEM[i].node[3]+1,
							ELEM[i].node[4]+1, ELEM[i].node[5]+1, ELEM[i].node[6]+1, ELEM[i].node[7]+1, ELEM[i].node[8]+1);
		}
		fprintf(stream,"E:%E  ni:%E  Ke:%E  Fii:%E  ci:%E  Edp_Fip1:%E  Edp_Fip2:%E  Fip:%E  Edp_cp1:%E  Edp_cp2:%E  cp:%E  Edp_Fir:%E  Fir:%E  Edp_cr:%E  cr:%E  beta:%E  Psir:%E\n",
			ELEM[i].matpars.E, ELEM[i].matpars.ni, ELEM[i].matpars.Ke, ELEM[i].matpars.Fii, ELEM[i].matpars.ci, ELEM[i].matpars.Edp_Fip1,
			ELEM[i].matpars.Edp_Fip2, ELEM[i].matpars.Fip, ELEM[i].matpars.Edp_cp1, ELEM[i].matpars.Edp_cp2, ELEM[i].matpars.cp, 
			ELEM[i].matpars.Edp_Fir, ELEM[i].matpars.Fir, ELEM[i].matpars.Edp_cr, ELEM[i].matpars.cr, ELEM[i].matpars.beta, ELEM[i].matpars.Psir );
	}
	fclose(stream);
}//Save_par
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void CreateSign( const char *name, long itval )
{
	FILE *stream;

	stream = fopen( name, "wt");    
    fprintf(stream,"Increment calculated.\n"); 
    fprintf(stream,"%ld\n", itval); 
	fclose(stream);
}//CreateSign
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//saving results for GID7 postprocessing
void Save4GID7( const char *name, int opt )
{
	FILE *stream;
	long k, i;
	COOR2 z;
	COOR3 t;

	if( opt == 0 )
	{
		stream = fopen( name, "wt");    
		fprintf(stream,"GiD Post Results File 1.0\n");

		//definitions of gauss points
		fprintf(stream,"\nGaussPoints \"T3\" ElemType Triangle\n");
		fprintf(stream,"Number of Gauss Points: 1\n");
		fprintf(stream,"Nodes not included\n");
		fprintf(stream,"Natural Coordinates: Internal\n");
		fprintf(stream,"End GaussPoints\n");
		
		fprintf(stream,"\nGaussPoints \"T6\" ElemType Triangle\n");
		fprintf(stream,"Number of Gauss Points: 3\n");
		fprintf(stream,"Nodes not included\n");
		fprintf(stream,"Natural Coordinates: Given\n");
		fprintf(stream,"0.0 0.5\n");
		fprintf(stream,"0.5 0.0\n");
		fprintf(stream,"0.5 0.5\n");
		fprintf(stream,"End GaussPoints\n");

		fprintf(stream,"\nGaussPoints \"Q4\" ElemType Quadrilateral\n");
		fprintf(stream,"Number of Gauss Points: 4\n");
		fprintf(stream,"Nodes not included\n");
		fprintf(stream,"Natural Coordinates: Internal\n");
		fprintf(stream,"End GaussPoints\n");

		fprintf(stream,"\nGaussPoints \"Q8Q9\" ElemType Quadrilateral\n");
		fprintf(stream,"Number of Gauss Points: 9\n");
		fprintf(stream,"Nodes not included\n");
		fprintf(stream,"Natural Coordinates: Internal\n");
		fprintf(stream,"End GaussPoints\n");
		
		//ranges
		fprintf(stream,"\nResultRangesTable \"stress\"\n");
		fprintf(stream,"-1000.0 - 1000.0: \"ok\"\n");
		fprintf(stream,"End ResultRangesTable\n");

		fprintf(stream,"\nResultRangesTable \"displacement\"\n");
		fprintf(stream,"-1.0 - 1.0: \"ok\"\n");
		fprintf(stream,"End ResultRangesTable\n");
		
		fprintf(stream,"\nResultRangesTable \"deformation\"\n");
		fprintf(stream,"-1.0 - 1.0: \"ok\"\n");
		fprintf(stream,"End ResultRangesTable\n");
		
		fclose(stream);		
	}
	else
	{
		stream = fopen( name, "at");    
		//results
		fprintf(stream,"\nResult \"Displacements\" \"%s\" %d Vector OnNodes \n", name, opt);
		fprintf(stream,"ResultRangesTable \"displacement\"\n" );
		fprintf(stream,"ComponentNames \"ux\", \"uy\", \"uz\"\n" );
		fprintf(stream,"Values\n" );		
		for( i=0; i<NNODE; i++)
		{
			z = readNODE( u_act, i );		
			fprintf(stream,"%d  %E  %E  %E\n", i+1, z.x, z.y, Zero);
		}
		fprintf(stream,"End Values\n");

		if( ELEM[0].typ == 3 )
		{
			fprintf(stream,"\nResult \"Deformations\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"T3\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"deformation\"\n" );
			fprintf(stream,"ComponentNames \"epsx\", \"epsy\", \"gamxy\", \"epsz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].eps[0];
				fprintf(stream,"%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, Zero);
			}
		}
		
		if( ELEM[0].typ == 4 )
		{
			fprintf(stream,"\nResult \"Deformations\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"Q4\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"deformation\"\n" );
			fprintf(stream,"ComponentNames \"epsx\", \"epsy\", \"gamxy\", \"epsz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].eps[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, Zero );
				for( k=1; k<4; k++ )
				{
					t = ELEM[i].eps[k];
					fprintf(stream,"    %E  %E  %E  %E\n", t.x, t.y, t.xy, Zero);
				}
			}
		}

		if( ELEM[0].typ == 6 )
		{
			fprintf(stream,"\nResult \"Deformations\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"T6\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"deformation\"\n" );
			fprintf(stream,"ComponentNames \"epsx\", \"epsy\", \"gamxy\", \"epsz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].eps[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, Zero );
				for( k=1; k<3; k++ )
				{
					t = ELEM[i].eps[k];
					fprintf(stream,"    %E  %E  %E  %E\n", t.x, t.y, t.xy, Zero);
				}
			}
		}

		if(( ELEM[0].typ == 8 ) || ( ELEM[0].typ == 9 ))
		{
			fprintf(stream,"\nResult \"Deformations\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"Q8Q9\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"deformation\"\n" );
			fprintf(stream,"ComponentNames \"epsx\", \"epsy\", \"gamxy\", \"epsz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].eps[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, Zero );
				for( k=1; k<9; k++ )
				{
					t = ELEM[i].eps[k];
					fprintf(stream,"    %E  %E  %E  %E\n", t.x, t.y, t.xy, Zero);
				}
			}
		}
		fprintf(stream,"End Values\n");
		
		if( ELEM[0].typ == 3 )
		{
			fprintf(stream,"\nResult \"Stresses\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"T3\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"stress\"\n" );
			fprintf(stream,"ComponentNames \"sigx\", \"sigy\", \"tauxy\", \"sigz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].sig[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, t.z );
			}
		}

		if( ELEM[0].typ == 4 )
		{
			fprintf(stream,"\nResult \"Stresses\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"Q4\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"stress\"\n" );
			fprintf(stream,"ComponentNames \"sigx\", \"sigy\", \"tauxy\", \"sigz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].sig[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, t.z );
				for( k=1; k<4; k++ )
				{
					t = ELEM[i].sig[k];
					fprintf(stream, "    %E  %E  %E  %E\n", t.x, t.y, t.xy, t.z );
				}
			}
		}

		if( ELEM[0].typ == 6 )
		{
			fprintf(stream,"\nResult \"Stresses\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"T6\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"stress\"\n" );
			fprintf(stream,"ComponentNames \"sigx\", \"sigy\", \"tauxy\", \"sigz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].sig[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, t.z );
				for( k=1; k<3; k++ )
				{
					t = ELEM[i].sig[k];
					fprintf(stream, "    %E  %E  %E  %E\n", t.x, t.y, t.xy, t.z );
				}
			}
		}

		if(( ELEM[0].typ == 8 ) || ( ELEM[0].typ == 9 ))
		{
			fprintf(stream,"\nResult \"Stresses\" \"%s\" %d PlainDeformationMatrix OnGaussPoints \"Q8Q9\"\n", name, opt);
			fprintf(stream,"ResultRangesTable \"stress\"\n" );
			fprintf(stream,"ComponentNames \"sigx\", \"sigy\", \"tauxy\", \"sigz\"\n" );
			fprintf(stream,"Values\n" );		
			for( i=0; i<NELEM; i++ )
			{
				t = ELEM[i].sig[0];
				fprintf(stream, "%d  %E  %E  %E  %E\n", i+1, t.x, t.y, t.xy, t.z );
				for( k=1; k<9; k++ )
				{
					t = ELEM[i].sig[k];
					fprintf(stream, "    %E  %E  %E  %E\n", t.x, t.y, t.xy, t.z );
				}
			}
		}
		fprintf(stream,"End Values\n");
		fclose(stream);
	}
}//Save4GID7
//-------------------------------------------------------------------------------------

/* END IOTools*/
#endif