/*
	File			:	FEMLib.CPP
	Identification	:	Library of functions and procedures for FEM
						Global variables & structures
	Author			:	P.E.Srokosz, 2006
	Purpose			:	Finite Elelements Method in Geomechanics
	Created			:	30.06.2006
	Last modified	:	08.07.2006
	Content			:	

	double deg2rad( double d )
	double rad2deg( double r )
	void KmZERO( void )
	void fZERO( void )
	void fRZERO( void )
	void uZERO( void )
	void BndRESET( void )
	void INITALL( void )
	double normV( double vec[MaxDim], long n )
	double maxV( double vec[MaxDim], long n )
	void loadNODE( long iNode, double Fx, double Fy )
	void uploadNODE( long iNode, double Fx, double Fy )
	void unloadNODE( long iNode, double Fx, double Fy )
	void addR2f( void )
	void updatefR( long iNode, double Fx, double Fy )
	void mkDE( double E, double ni, double Ke )
	void mkDU( double Ke )
	void saveCONSTRAINT( long iNode, double ux, double uy )
	void dispNODE( long iNode, double ux, double uy )
	void Constraints( void )
	void penaltyNODE( long iNode, double ux, double uy )
	COOR2 readNODE( double vec[MaxDim], long iNode )
	double trunc( double x )
	void SCALEEQ( void )
	int GAUSS( void )
	int KmINVERSE( void )

*/

#ifndef _srokosz_FEMLib_cpp
#define _srokosz_FEMLib_cpp

#include <iso646.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <stdio.h>

/* BEGIN */

/**********************************
 constants, variables & structures
 **********************************/

#include "FEMVars.h"

PARAMS ELEM[MaxElem];

COOR2 NODE[MaxNode];

BOUNDS Bnd[MaxBound];

long NELEM, NNODE, NDIM, NBND, BMPheight, BMPwidth, totalGP, FastDT, IterCtrl, RetCtrl;

double Km[MaxDim][MaxDim], DE[4][4], DU[4][4], STRESS[4], STRAIN[4], f[MaxDim], u[MaxDim];
double u_act[MaxDim], fResidual[MaxDim], delta, Fsafety, Vcrit, u_incr[MaxDim];
double DT, u_elas[MaxDim], NRRelax, f_ext[MaxDim], f_total[MaxDim], fR_copy[MaxDim];

/*
Old order

GAUSSPOINTS GPT6[3] = {{ Half, Half, WT6 }, { Half, Zero, WT6 }, { Zero, Half, WT6 }};
GAUSSPOINTS GPQ4[4] = {{ NSqrtI3, SqrtI3, One }, { SqrtI3, SqrtI3, One }, { NSqrtI3, NSqrtI3, One }, { SqrtI3, NSqrtI3, One }};
GAUSSPOINTS GPQ8[9] = {{ NSqrtI5, SqrtI5, WG4 }, { Zero, SqrtI5, WG2 }, { SqrtI5, SqrtI5, WG4 },
						{ NSqrtI5, Zero, WG2 }, { Zero, Zero, WG1 }, { SqrtI5, Zero, WG2 }, 
						{ NSqrtI5, NSqrtI5, WG4 }, { Zero, NSqrtI5, WG2 }, { SqrtI5, NSqrtI5, WG4 }};
*/

//New order

GAUSSPOINTS GPT6[3] = {{ Half, Zero, WT6 }, { Half, Half, WT6 }, { Zero, Half, WT6 }};
GAUSSPOINTS GPQ4[4] = {{ NSqrtI3, NSqrtI3, One }, { SqrtI3, NSqrtI3, One }, { SqrtI3, SqrtI3, One }, { NSqrtI3, SqrtI3, One }};
GAUSSPOINTS GPQ8[9] = {{ NSqrtI5, NSqrtI5, WG4 }, { SqrtI5, NSqrtI5, WG2 }, { SqrtI5, SqrtI5, WG4 },
						{ NSqrtI5, SqrtI5, WG2 }, { Zero, NSqrtI5, WG1 }, { SqrtI5, Zero, WG2 }, 
						{ Zero, SqrtI5, WG4 }, { NSqrtI5, Zero, WG2 }, { Zero, Zero, WG4 }};

double INCREMENT, dSTRESS[4];
unsigned char BMPData[20000000];
MATPAR PAR[256];
MATPAR MATPAR_ZERO = {	Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero, Zero };
COOR3 COOR3_ZERO = { Zero, Zero, Zero, Zero };
COOR2 COOR2_ZERO = { Zero, Zero };
long Nmonitor, Iter; //total number of nodes to monitor
long NODEmonitor[MaxBound]; //nodes to monitor
long MatIdentType; //material identification type in element: 1 averaging of nodes, 2 on center point


//-------------------------------------------------------------------------------------
double deg2rad( double d )
{
	double r;
	
	r = d * M_PI / (double) 180.0;
	return r;
}//deg2rad
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
double rad2deg( double r )
{
	double d;
	
	d = r * (double) 180.0 / M_PI;
	return d;
}//rad2deg
//-------------------------------------------------------------------------------------

/******************************
        Initializations
 ******************************/

//-------------------------------------------------------------------------------------
void KmZERO( void )
{
	long i, j;

	for( i=0; i<MaxDim; i++ )
	{
		for( j=0; j<MaxDim; j++ )
			Km[i][j] = Zero;
	}
}//KmZERO
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void fZERO( void )
{
	long i;

	for( i=0; i<MaxDim; i++ ) f[i] = Zero;
}//fZERO
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void fRZERO( void )
{
	long i;

	for( i=0; i<MaxDim; i++ ) fResidual[i] = Zero;
}//fRZERO
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void uZERO( void )
{
	long i;

	for( i=0; i<MaxDim; i++ ) u[i] = Zero;
}//uZERO
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void BndRESET( void )
{
	long i;

	for( i=0; i<MaxBound; i++ )
	{
		Bnd[i].node = 0;
		Bnd[i].ux = Free;
		Bnd[i].uy = Free;
	}
	NBND = 0;
}//BndRESET
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void INITALL( void )
{
	long i, iElem;

	KmZERO( );
	fZERO( );
	uZERO( );
	fRZERO( );
	BndRESET( );
	NELEM = 0L;
    NNODE = 0L;
	NDIM = 0L;
	for( iElem=0L; iElem<MaxElem; iElem++)
	{
		ELEM[iElem].matpars = MATPAR_ZERO;
		for(i=0L; i<9L; i++)
		{
			ELEM[iElem].Fmc[i] = NOne;
			ELEM[iElem].eps[i] = COOR3_ZERO;
			ELEM[iElem].sig[i] = COOR3_ZERO;
			ELEM[iElem].dsig[i] = COOR3_ZERO;
			ELEM[iElem].deps[i] = COOR3_ZERO;
			ELEM[iElem].depsvp[i] = COOR3_ZERO;
			ELEM[iElem].dsigvp[i] = COOR3_ZERO;
			ELEM[iElem].dQ[i] = COOR3_ZERO;
			ELEM[iElem].Gauss[i] = COOR2_ZERO;
			ELEM[iElem].u[i] = Zero;
			ELEM[iElem].plas[i] = 0L;
			ELEM[iElem].Edp[i]  = Zero;
		}
	}
}//INITALL
//-------------------------------------------------------------------------------------

/******************************
         Data preparing
 ******************************/

//-------------------------------------------------------------------------------------
//(Euclides) norm of vector
double normV( double vec[MaxDim], long n )
{
	double nV;
	long row;
	
	nV = Zero;
	for( row=0; row<n; row++ )
	{
		nV += vec[row] * vec[row];
	}
	nV = sqrt(nV);
	return nV;

}//normV
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Max norm of vector
double maxV( double vec[MaxDim], long n )
{
	double nV;
	long row;
	
	nV = Zero;
	for( row=0; row<n; row++ )
	{
		if( nV < fabs(vec[row]) ) nV = fabs(vec[row]);
	}
	return nV;

}//maxV
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void mkNODE( long iNode, double x, double y )
{
	NNODE++;
	NODE[iNode].x = x;
	NODE[iNode].y = y;
	NDIM = 2 * NNODE;
}//mkNODE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//forces on nodes
void loadNODE( long iNode, double Fx, double Fy )
{
	f[2 * iNode] = Fx;
	f[2 * iNode + 1] = Fy;
}//loadNODE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//forces on nodes
void uploadNODE( long iNode, double Fx, double Fy )
{
	f[2 * iNode] += Fx;
	f[2 * iNode + 1] += Fy;
}//uploadNODE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//unload forces on nodes
void unloadNODE( long iNode, double Fx, double Fy )
{
	f[2 * iNode] -= Fx;
	f[2 * iNode + 1] -= Fy;
}//unloadNODE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//add forces on nodes
void addR2f( void )
{
	long i;

	for( i=0; i<NDIM; i++ )
	{
		f[i] += fResidual[i];
	}
}//addB2f
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
void updatefR( long iNode, double Fx, double Fy )
{
	
	fResidual[2 * iNode] += Fx;
	fResidual[2 * iNode + 1] += Fy;

}//savefR
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//forms elastic matrix DE - plain strain
void mkDE( double E, double ni, double Ke )
{
   	double uni;

	uni = E / (( One + ni ) * ( One - Two * ni ));
	DE[0][0] = uni * ( One - ni ) + Ke;
	DE[0][1] = uni * ni + Ke;
	DE[0][2] = Zero;
	DE[0][3] = uni * ni + Ke;
	DE[1][0] = uni * ni + Ke;
	DE[1][1] = uni * ( One - ni ) + Ke;
	DE[1][2] = Zero;
	DE[1][3] = uni * ni + Ke;
	DE[2][0] = Zero;
	DE[2][1] = Zero;
	DE[2][2] = uni * (One - Two * ni) * Half;
	DE[2][3] = Zero;
	DE[3][0] = uni * ni + Ke;
	DE[3][1] = uni * ni + Ke;
	DE[3][2] = Zero;
	DE[3][3] = uni * ( One - ni ) + Ke;
}//mkDE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//forms elastic matrix DU - pore pressure
void mkDU( double Ke )
{
	DU[0][0] = Ke;
	DU[0][1] = Ke;
	DU[0][2] = Zero;
	DU[0][3] = Ke;
	DU[1][0] = Ke;
	DU[1][1] = Ke;
	DU[1][2] = Zero;
	DU[1][3] = Ke;
	DU[2][0] = Zero;
	DU[2][1] = Zero;
	DU[2][2] = Zero;
	DU[2][3] = Zero;
	DU[3][0] = Ke;
	DU[3][1] = Ke;
	DU[3][2] = Zero;
	DU[3][3] = Ke;
}//mkDU
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//prescribed displacements of nodes - classic method
void saveCONSTRAINT( long iNode, double ux, double uy )
{
	NBND++;
	Bnd[NBND].node = iNode;
	Bnd[NBND].ux = ux;
	Bnd[NBND].uy = uy;

}//saveCONSTRAINT
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//prescribed displacements of nodes - classic method
void dispNODE( long iNode, double ux, double uy )
{
	long row, i;

	NBND++;
	Bnd[NBND].node = iNode;
	Bnd[NBND].ux = ux;
	Bnd[NBND].uy = uy;

	row = 2L * iNode;
	
	if( fabs(ux)<Free )
	{		
		f[row] = Zero;
		for( i=0L; i<NDIM; i++ )
		{
			f[i] -= Km[i][row] * ux;
			Km[i][row] = Zero;
		}
		Km[row][row] = NOne;
	}

	if( fabs(uy)<Free )
	{
		row++;
		f[row] = Zero;
		for( i=0L; i<NDIM; i++ )
		{
			f[i] -= Km[i][row] * uy;
			Km[i][row] = Zero;
		}
		Km[row][row] = NOne;
	}
}//dispNODE
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//ReActivate constraints - classic method
void Constraints( void )
{
	long row, i, j;

	if( NBND>0 )
	{
		for( j=1L; j<=NBND; j++ ) 
		{
			row = 2L * Bnd[j].node;
	
			if( fabs(Bnd[j].ux)<Free )
			{		
				f[row] = Zero;
				for( i=0L; i<NDIM; i++ )
				{
					f[i] -= Km[i][row] * Bnd[j].ux;
					Km[i][row] = Zero;
				}
				Km[row][row] = NOne;
			}

			if( fabs(Bnd[j].uy)<Free )
			{
				row++;
				f[row] = Zero;
				for( i=0L; i<NDIM; i++ )
				{
					f[i] -= Km[i][row] * Bnd[j].uy;
					Km[i][row] = Zero;
				}
				Km[row][row] = NOne;
			}
		}
	}
}//Constraints
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//prescribed displacements of nodes - penalty method
void penaltyNODE( long iNode, double ux, double uy )
{
	long row;

	NBND++;
	Bnd[NBND].node = iNode;
	Bnd[NBND].ux = ux;
	Bnd[NBND].uy = uy;

	row = 2 * iNode;
	
	if( fabs(ux)<Free )
	{		
		Km[row][row] += (double) 1.0E+20;
		f[row] = ux * Km[row][row];
	}

	if( fabs(uy)<Free )
	{
		row++;
		Km[row][row] += (double) 1.0E+20;
		f[row] = uy * Km[row][row];
	}
}//penaltyNODE
//-------------------------------------------------------------------------------------

/******************************
      Retrieving results
 ******************************/

//-------------------------------------------------------------------------------------
COOR2 readNODE( double vec[MaxDim], long iNode )
{
	COOR2 v;
	
	v.x = vec[2 * iNode];
	v.y = vec[2 * iNode + 1];
	return v;
}//readNODE
//-------------------------------------------------------------------------------------

/******************************
 Linear equation system solver
 ******************************/

//-------------------------------------------------------------------------------------
//pascal ROUND function
//round
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//pascal TRUNC function
double trunc( double x )
{
	if( x < Zero )
	{
		return ceil( x );
	}
	else
	{
		return floor( x );
	}
}//trunc
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Scalling matrix Km and vector f: for GAUSS elimination process
void SCALEEQ( void )
{
	long i, j;
    double LN2, S, S1;

	LN2 = log(Two);

	for( i=0; i<NDIM; i++ )
	{  
		S = Zero;
		//Norm of row i
		for( j=0; j<NDIM; j++ ) S += fabs(Km[i][j]); 
		//Scalling coefficient for i-row
		S1 = log(S) / LN2 + One;
		if( S<Half or S>One )
		{
			if( S1>=Zero )
			{
				S = exp( NOne * trunc(S1) * LN2 );
			}
			else
			{
				if( (S1 - trunc(S1)) == Zero )
				{
					S = exp( NOne * trunc(S1) * LN2 );
				}
                else
				{
					S = exp( NOne * trunc(S1 - One) * LN2 );
				}
			}
			for( j=0; j<NDIM; j++ )
			{
				Km[i][j] *= S;
			}
			f[i] *= S;
		}//S<0.5 or S>1.0
	}//i++
}//SCALEEQ
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Gauss elimination procedure
int GAUSS( void )
{
	long i, j, k;
    double T, S;

    //Scalling matrix Km and vector f (works on global data)
	SCALEEQ( );

	//Konstrukcja ciagu macierzy Km(i) oraz ciagu wektorow f(i)
	for( i=0; i<NDIM; i++ )
	{
		//Wybor elementu glownego
		T = fabs(Km[i][i]); 
		k = i;
		for( j=(i+1); j<NDIM; j++ )
		{
			if( fabs(Km[j][i])>T )
			{
				T = fabs(Km[j][i]); 
				// Zamiana wiersza k-tego z i-tym
				k = j; 
			}
		}
		//Warunek przerwania poszukiwania elementu glownego
		//Warunek nieistnienia rozwiazania rownania
 		if( T<EPS )
		{
			return 1;
		}
    
		if( i==k )
		{
			T = Km[i][i];
		}
		else
		{
             //Zamiana elementu k-tego z i-tym wektora f
             T = f[k]; f[k] = f[i]; f[i] = T;
             // Zamiana wiersza k-tego z i-tym macierzy Km
             for( j=(NDIM-1); j>=i; j-- )
             {
				T = Km[k][j]; Km[k][j] = Km[i][j]; Km[i][j] = T;
             }
		}
		T = One / T; Km[i][i] = T;
		for( j=(i+1); j<NDIM; j++ )
		{
			S = NOne * Km[j][i] * T;         
			f[j] += f[i] * S;    
			for( k=(i+1); k<NDIM; k++ )
				Km[j][k] += Km[i][k] * S;
		}
	}//i=0 to ndim-1
    
	//Rozwiazanie ukladu trojkatnego metoda postepowania wstecz
    for( i=(NDIM-1); i>=0; i-- )
	{
		T = f[i];
		for( j=(i+1); j<NDIM; j++ )
			T -= Km[i][j] * u[j];
		u[i] = T * Km[i][i];
	}
	
	return 0;
}//GAUSS
//-------------------------------------------------------------------------------------

//-------------------------------------------------------------------------------------
//Inversion of stiffness matrix
int KmINVERSE( void )
{
	long i, j, l, k, M[MaxDim];
    double maxKm, d, e;

	for( i=0L; i<NDIM; i++ )
	{
		//Czesciowy wybor elementu glownego
		maxKm = Zero;
		for( j=i; j<NDIM; j++ )
		{
			d = Km[j][i];
			if( fabs(maxKm) < fabs(d) ) 
			{
				maxKm = d; 
				k = j;
			}
		}
		if( fabs(maxKm) < EPS )	return 1;
		
		//Zapisywanie wskaznikow wierszy wystepowania elementu
		//ekstremalnego w i-tej iteracji w postaci wektora M[i]
		M[i] = k;
		Km[k][i] = One;
		for( j=0L; j<NDIM; j++ )
		{
			//Przestawienie i-tego wiersza z k-tym 
			d = Km[k][j] / maxKm;
			Km[k][j] = Km[i][j];
			Km[i][j] = d;
		}
		
		//Generacja ciagu macierzy wg wzoru rekurencyjnego
		for( j=0; j<NDIM; j++ )
		{
			if( j != i )
			{
				d = Km[j][i];
				Km[j][i] = Zero;
				for( l=0L; l<NDIM; l++ )
				{
					e = d * Km[i][l];
					Km[j][l] = Km[j][l] - e;
				}
			}
		}
	}
	
	//Przestawianie kolumn macierzy zgodnie z wektorem wskaznikow M[i]
	for( i=(NDIM-1L); i>=0; i-- )
	{
		k = M[i];
		if( k!=i )
		{
			for( j=0L; j<NDIM; j++ )
			{
				d = Km[j][i];
				Km[j][i] = Km[j][k];
				Km[j][k] = d;
			}
		}
	}
	return 0;
}//KmINVERSE
//-------------------------------------------------------------------------------------


#endif

/* END */